% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/runNestedApprox.R
\name{runNestedApprox}
\alias{runNestedApprox}
\title{Run a nested approximation, returning a summary object with default inference}
\usage{
runNestedApprox(
  approx,
  quantiles = c(0.025, 0.25, 0.5, 0.75, 0.975),
  originalScale = TRUE,
  improve1d = TRUE,
  nSamplesLatents = 0,
  nSamplesParams = 0
)
}
\arguments{
\item{approx}{a compiled or uncompiled nestedApprox object created by \code{buildNestedApprox}.}

\item{quantiles}{numeric vector of quantiles to estimate for each parameter. Default is \code{c(0.025, 0.25, 0.5, 0.75, 0.975)}.}

\item{originalScale}{logical; if \code{TRUE}, report results on the original scales of the parameters and latent nodes. 
Default is \code{TRUE}.}

\item{improve1d}{logical; if \code{TRUE} and there is a single parameter, improve the estimate of the estimate of marginal
distribution for the marginal by directly using the Laplace/AGHQ approximate marginal distribution rather than the asymmetric
Gaussian approximation. Default is \code{TRUE}.}

\item{nSamplesLatents}{number of samples of the latent nodes to draw. Default is 0.}

\item{nSamplesParams}{number of samples of the parameter nodes to draw. Default is 0.}
}
\value{
An object of class \code{approxSummary} containing initial results that can be used to carry out further inference.
}
\description{
Uses a nested approximation (compiled or uncompiled) returned from \code{buildNestedApprox}) 
to do default inference and return a summary object that can be used for additional tailored inference.
It estimates marginal distributions for parameters (quantiles and expectations), and can 
optionally return posterior samples for the latent nodes and parameters
}
\details{
This is the main user interface for running a nested approximation. It carries out
default inference and then returns a summary object that can be used for further inference
by calling methods on the summary object, as seen in the examples (or running the equivalent 
function calls with the first argument being the summary object).
}
\section{Methods available for object of class \code{approxSummary}}{


Once the default inference has been run, inference can then be improved by calling different available methods within the returned object.
Each method is explained in detail in their documentation, but the user may choose the following options:

\itemize{
    \item \code{setParamGrid}. Allows the user to change the parameter grid used in the nested approximation.         
    \item \code{improveParamMarginals}. Improve univariate parameter marginals using grid-based quadrature.
    \item \code{calcMarginalLogLikImproved}. Calculate improved marginal log-likelihood using grid-based quadrature.
    \item \code{sampleParams}. Sample from the parameter posterior distribution.
    \item \code{sampleLatents}. Sample from the posterior distribution of the latent nodes.
    \item \code{qmarginal}. Compute quantiles for a parameter.
    \item \code{dmarginal}. Compute marginal density values for a parameter.
    \item \code{rmarginal}. Draw random samples from the marginal posterior of a parameter.
    \item \code{emarginal}. Compute the expectation of a function of a parameter under the marginal posterior distribution.
    \item \code{plotMarginal}. Plot the marginal posterior for a parameter.
  }
}

\examples{
data(penicillin, package="faraway")
code <- nimbleCode({
    for(i in 1:n) {
        mu[i] <- inprod(b[1:nTreat], x[i, 1:nTreat]) + re[blend[i]]
        y[i] ~ dnorm(mu[i], tau = Tau)
    }
    # Priors corresponding simply to INLA defaults and not being recommended.
    # Instead consider uniform or half-t distributions on the standard deviation scale
    # or penalized complexity priors.
    Tau ~ dgamma(1, 5e-05)
    Tau_re ~ dgamma(1, 5e-05)
    for( i in 1:nTreat ){ b[i] ~ dnorm(0, tau = 0.001) }
    for( i in 1:nBlend ){ re[i] ~ dnorm(0, tau = Tau_re) }
})
X <- model.matrix(~treat, data = penicillin)
data = list(y = penicillin$yield)
constants = list(nTreat = 4, nBlend = 5, n = nrow(penicillin),
                 x = X, blend = as.numeric(penicillin$blend))
inits <- list(Tau = 1, Tau_re = 1, b = c(mean(data$y), rep(0,3)), re = rep(0,5))

model <- nimbleModel(code, data = data, constants = constants,
                 inits = inits, buildDerivs = TRUE)
approx <- buildNestedApprox(model = model)

\donttest{
comp_model <- compileNimble(model)
comp_approx <- compileNimble(approx, project = model)
result <- runNestedApprox(comp_approx)
# Improve marginals for a parameter node using AGHQ.
result$improveParamMarginals(nodes = 'Tau_re', nMarginalGrid = 9)
# Specify other quantiles of interest.
result$qmarginal('Tau_re', quantiles = c(.05, .95))
# Compute other expectations of interest, here the mean on the standard deviation scale
result$emarginal('Tau_re', function(x) 1/sqrt(x))

# Sample from the approximate posterior for the latent nodes.
latent_sample <- result$sampleLatents(n = 1000)
# For joint inference on parameters, sample from the approximate posterior for the parameters.
param_sample <- result$sampleParams(n = 1000)
}

}
\author{
Christopher Paciorek
}
