message("Warning: over 8200 tests. Takes a couple of minutes to run.")
delta1 <- 1.0e-8
epsilon <- 1.0e-3

n <- vision_data
M <- nrow(n)
c_value <- 1.0
alpha <- rep(1.0, M)
delta <- McCullagh_initialize_delta(n)
delta_vec <- McCullagh_initialize_delta_vec(n)
psi <- McCullagh_initialize_psi(n, delta, alpha, c_value)


test_that("McCullagh_compute_pij_qij", {
  M <- nrow(psi)
  c <- c_value
  pqij <- McCullagh_compute_cumulatives(psi, delta, alpha, c)
  pij <- pqij$pij
  qij <- pqij$qij

  pi <- McCullagh_compute_pi(psi, delta, alpha, c)

  for (i in 1:M) {
    for (j in 1:M) {
      expect_true(pi[i, j] > 0.0, info=paste("pi <=0.0", i, j, pi[i, j], sep=" "))
    }
  }

  expect_true(abs(psi[1, 1] - pi[1, 1]) <= epsilon, info="1 1")
  expect_true(abs(psi[2, 2] - pi[2, 2]) <= epsilon, info="2 2")
  expect_true(abs(psi[3, 3] - pi[3, 3]) <= epsilon, info="3 3")
  expect_true(abs(psi[4, 4] - pi[4, 4]) <= epsilon, info="4 4")
  expect_true(abs(pij[1, 4] - pi[1, 4]) <= epsilon, info="1 4")
  expect_true(abs(qij[1, 4] - pi[4, 1]) <= epsilon, info="4 1")
  expect_true(abs(pij[1, 3] - pij[1, 4] - pi[1, 3]) <= epsilon, info="1 3")
  expect_true(abs(pij[1, 2] - pij[1, 3] - pi[1, 2]) <= epsilon, info="1 2")
  expect_true(abs(qij[1, 3] - qij[1, 4] - pi[3, 1]) <= epsilon, info="3 1")
  expect_true(abs(qij[1, 2] - qij[1, 3] - pi[2, 1]) <= epsilon, info="2 1")
  expect_true(abs(pij[3, 4] - pij[2, 4] - pi[3, 4]) <= epsilon, info="3 3")
  expect_true(abs(qij[3, 4] - qij[2, 4] - pi[4, 3]) <= epsilon, info="4 2")
  expect_true(abs(pij[2, 4] - pij[1, 4] - pi[2, 4]) <= epsilon, info="2 3")
  expect_true(abs(qij[2, 4] - qij[1, 4] - pi[4, 2]) <= epsilon, info="4 2")
  expect_true(abs(pij[2, 3] - pij[1, 3] - pij[2, 4] + pij[1, 4] - pi[2, 3]) <= epsilon, info="2 3")
  expect_true(abs(qij[2, 3] - qij[1, 3] - qij[2, 4] + qij[1, 4] - pi[3, 2]) <= epsilon, info="3 2")
}
)


test_McCullagh_compute_generalized_pij_qij <- function(psi, delta_vec, alpha, c=1.0) {
  M <- nrow(psi)
  pqij <- McCullagh_compute_generalized_cumulatives(psi, delta_vec, alpha, c)
  pij <- pqij$pij
  qij <- pqij$qij

  pi <- McCullagh_compute_generalized_pi(psi, delta_vec, alpha, c)

  for (i in 1:M) {
    for (j in 1:M) {
      expect_true(pi[i, j] > 0.0, info=paste("pi <=0.0", i, j, pi[i, j], sep=" "))
    }
  }

  expect_true(abs(psi[1, 1] - pi[1, 1]) <= epsilon, info="1 1")
  expect_true(abs(psi[2, 2] - pi[2, 2]) <= epsilon, info="2 2")
  expect_true(abs(psi[3, 3] - pi[3, 3]) <= epsilon, info="3 3")
  expect_true(abs(psi[4, 4] - pi[4, 4]) <= epsilon, info="4 4")
  expect_true(abs(pij[1, 4] - pi[1, 4]) <= epsilon, info="1 4")
  expect_true(abs(qij[1, 4] - pi[4, 1]) <= epsilon, info="4 1")
  expect_true(abs(pij[1, 3] - pij[1, 4] - pi[1, 3]) <= epsilon, info="1 3")
  expect_true(abs(pij[1, 2] - pij[1, 3] - pi[1, 2]) <= epsilon, info="1 2")
  expect_true(abs(qij[1, 3] - qij[1, 4] - pi[3, 1]) <= epsilon, info="3 1")
  expect_true(abs(qij[1, 2] - qij[1, 3] - pi[2, 1]) <= epsilon, info="2 1")
  expect_true(abs(pij[3, 4] - pij[2, 4] - pi[3, 4]) <= epsilon, info="3 3")
  expect_true(abs(qij[3, 4] - qij[2, 4] - pi[4, 3]) <= epsilon, info="4 2")
  expect_true(abs(pij[2, 4] - pij[1, 4] - pi[2, 4]) <= epsilon, info="2 3")
  expect_true(abs(qij[2, 4] - qij[1, 4] - pi[4, 2]) <= epsilon, info="4 2")
  expect_true(abs(pij[2, 3] - pij[1, 3] - pij[2, 4] + pij[1, 4] - pi[2, 3]) <= epsilon, info="2 3")
  expect_true(abs(qij[2, 3] - qij[1, 3] - qij[2, 4] + qij[1, 4] - pi[3, 2]) <= epsilon, info="3 2")
}


test_that("McCullagh_compute_weighted_pi", {
  M <- nrow(psi)
  c <- c_value

  pqij <- McCullagh_compute_cumulatives(psi, delta, alpha, c)
  pij <- pqij$pij
  qij <- pqij$qij

  pi <- McCullagh_compute_pi(psi, delta, alpha, c)
  for (i in 1:M) {
    for (j in 1:M) {
      weights <- McCullagh_extract_weights(i, j, M)
      w_psi <- weights$w_psi
      w_pij <- weights$w_pij
      w_qij <- weights$w_qij
      if (i == j) {
        w <- w_psi
        p_values <- psi
      } else if (i < j) {
        w <- w_pij
        p_values <- pij
      } else {
        w <- w_qij
        p_values <- qij
      }

      pi_weighted <- 0.0
      for (a in 1:M) {
        for (b in 1:M) {
          if (i <= j) {
            wgt = w[a, b]
            pi_weighted <- pi_weighted + wgt * p_values[a, b]
          } else if (i > j) {
            wgt = w[b, a]
            pi_weighted <- pi_weighted + wgt * p_values[b, a]
          }
        }
      }
      expect_true(abs(pi_weighted - pi[i, j]) <= epsilon, info=paste(i, j, pi[i,j], pi_weighted, sep=" "))
    }
  }
}
)


test_that("McCullagh_derivative_pij_qij_wrt_psi", {
  M <- nrow(psi)
  c <- c_value

  pqij <- McCullagh_compute_cumulatives(psi, delta, alpha, c)
  pij <- pqij$pij
  qij <- pqij$qij

  for (i in 1:M) {
    for (j in 1:M) {
      if (i < j) {
        expect_true(abs(pij[i, j] - McCullagh_pij_qij(i, j, psi, delta, alpha, c)) <= epsilon,
            info=paste(i, j, pij[i, j], McCullagh_pij_qij(i, j, psi, delta, alpha), sep=" "))

      } else if (j < i) {
        expect_true(abs(qij[j, i] - McCullagh_pij_qij(i, j, psi, delta, alpha, c)) <= epsilon,
            info=paste(i, j, pij[i, j], McCullagh_pij_qij(i, j, psi, delta, alpha), sep=" "))
      }

      for (h in 1:M) {
        for (k in 1:M) {
          psi[h, k] <- psi[h, k] - delta1
          psi[k, h] <- psi[h, k]
          f0 <- McCullagh_pij_qij(i, j, psi, delta, alpha, c)

          psi[h, k] <- psi[h,k] + 2.0 * delta1
          psi[k, h] <- psi[h, k]
          f1 <- McCullagh_pij_qij(i, j, psi, delta, alpha, c)

          psi[h, k] <- psi[h, k] - delta1
          psi[k, h] <- psi[h, k]

          der <- ((pij[i, j] - f0) / delta1 + (f1 - pij[i, j]) / delta1) / 2.0
          deriv <- McCullagh_derivative_pij_wrt_psi(i, j, h, k, delta, alpha)
          expect_true(abs(der - deriv) <= epsilon,
              info=paste(i, j, h, k, der, deriv, sep=" "))
        }
      }
    }
  }
}
)


test_that("McCullagh_derivative_pij_qij_wrt_delta", {
  M <- nrow(psi)
  c <- c_value

  pqij <- McCullagh_compute_cumulatives(psi, delta, alpha, c)
  pij <- pqij$pij
  qij <- pqij$qij

  for (i in 1:M) {
    for (j in 1:M) {
      if (i < j) {
        expect_true(abs(pij[i, j] - McCullagh_pij_qij(i, j, psi, delta, alpha, c)) <= epsilon,
            info=paste(i, j, pij[i, j], McCullagh_pij_qij(i, j, psi, delta, alpha), sep=" "))
      } else if (j < i) {
        expect_true(abs(qij[j, i] - McCullagh_pij_qij(i, j, psi, delta, alpha, c)) <= epsilon,
            info=paste(i, j, qij[j, i], McCullagh_pij_qij(i, j, psi, delta, alpha), sep = " "))
      }

      delta <- delta - delta1
      f0 <- McCullagh_pij_qij(i, j, psi, delta, alpha, c)

      delta <- delta + 2.0 * delta1
      f1 <- McCullagh_pij_qij(i, j, psi, delta, alpha, c)

      delta = delta - delta1

      if (i < j) {
        der <- ((pij[i, j] - f0) / delta1 + (f1 - pij[i, j]) / delta1) / 2.0
      } else if (j < i) {
        der = ((qij[j, i] - f0) / delta1 + (f1 - qij[j, i]) / delta1) / 2.0
      } else {
        der = 0.0
      }
      deriv <- McCullagh_derivative_pij_wrt_delta(i, j, psi, delta, alpha, c)
      expect_true(abs(der - deriv) <= epsilon, info=paste(i, j, der, deriv, sep=" "))
    }
  }
}
)


test_that("McCullagh_derivative_pij_qij_wrt_alpha", {
  M <- nrow(psi)
  c <- c_value

  pqij <- McCullagh_compute_cumulatives(psi, delta, alpha, c)
  pij <- pqij$pij
  qij <- pqij$qij

  for (i in 1:M) {
    for (j in 1:M) {
      f <- McCullagh_pij_qij(i, j, psi, delta, alpha, c)
      if (i < j) {
        expect_true(abs(pij[i, j] - f) <= epsilon, info=paste(i, j, pij[i, j], f, sep=" "))
      } else if (j < i) {
        expect_true(abs(qij[j, i] - f) <= epsilon, info=paste(i, j, qij[j, i], f, sep=" "))
      }
      for (i1 in 2:(M - 1)) {
        alpha[i1] <- alpha[i1] - delta1
        f0 <- McCullagh_pij_qij(i, j, psi, delta, alpha, c)

        alpha[i1] <- alpha[i1] + 2.0 * delta1
        f1 <- McCullagh_pij_qij(i, j, psi, delta, alpha, c)

        alpha[i1] <- alpha[i1] - delta1

        der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
        deriv <- McCullagh_derivative_pij_wrt_alpha(i, j, i1, psi, delta, alpha)
        expect_true(abs(der - deriv) <= delta1, info=paste(i, j, i1, der, deriv))
      }
    }
  }
}
)


test_that("McCullagh_derivative_pij_qij_wrt_c", {
  M <- nrow(psi)
  c <- c_value

  pqij <- McCullagh_compute_cumulatives(psi, delta, alpha, c)
  pij <- pqij$pij
  qij <- pqij$qij

  # i and j index pi
  for (i in 1:M) {
    for (j in 1:M) {
      if (i < j) {
        expect_true(abs(pij[i, j] - McCullagh_pij_qij(i, j, psi, delta, alpha, c)) <= epsilon,
                    info=paste(i, j, pij[i, j], McCullagh_pij_qij("pij", i, j, psi, delta, alpha), sep=" "))
      } else if (j < i) {
        expect_true(abs(qij[j, i] - McCullagh_pij_qij(i, j, psi, delta, alpha, c)) <= epsilon,
                    info=paste(i, j, qij[j, i], McCullagh_pij_qij("qij", i, j, psi, delta, alpha), sep = " "))
      }

      c <- c - delta1
      f0 <- McCullagh_pij_qij(i, j, psi, delta, alpha, c)

      c <- c + 2.0 * delta1
      f1 <- McCullagh_pij_qij(i, j, psi, delta, alpha, c)

      c = c - delta1

      if (i < j) {
        der <- ((pij[i, j] - f0) / delta1 + (f1 - pij[i, j]) / delta1) / 2.0
      } else if (j < i) {
        der = ((qij[j, i] - f0) / delta1 + (f1 - qij[j, i]) / delta1) / 2.0
      } else {
        der = 0.0
      }
      deriv <- McCullagh_derivative_pij_wrt_c(i, j, psi, delta, alpha, c)
      expect_true(abs(der - deriv) <= epsilon, info=paste("deriv", i, j, der, deriv, sep=" "))
    }
  }
}
)


test_that("McCullagh_derivative_pij_qij_wrt_delta_vec", {
  M <- nrow(psi)
  c <- c_value

  for (k in 1:(M - 1)) {
    for (i in 1:(M - 1)) {
      for (j in 1:M) {
        result <- McCullagh_compute_generalized_cumulatives(psi, delta_vec, alpha, c)
        f <- result$pij
        if (i > j) {
          f <- result$qij
        }

        delta_vec[k] <- delta_vec[k] - delta1
        result0 <- McCullagh_compute_generalized_cumulatives(psi, delta_vec, alpha, c)
        f0 <- result0$pij
        if (i > j) {
          f0 <- result0$qij
        }

        delta_vec[k] <- delta_vec[k] + 2.0 * delta1
        result1 <- McCullagh_compute_generalized_cumulatives(psi, delta_vec, alpha, c)
        f1 <- result1$pij
        if (i > j) {
          f1 <- result1$qij
        }

        der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
        deriv <- McCullagh_derivative_pij_wrt_delta_vec(i, j, k, psi, delta_vec, alpha, c=1.0)
        if (i <= j) {
          expect_true(abs(der[i, j] - deriv) <= epsilon,
                      info=paste("generalized pij", i, j, k, der[i, j], deriv))
        } else {
          expect_true(abs(der[j, i] - deriv) <= epsilon,
                      info=paste("generalized pij", i, j, k, der[j, i], deriv))
        }
      }
    }
  }
}
)


test_that("McCullagh_derivative_pi_wrt_psi", {
  M <- nrow(psi)
  c <- c_value

  for (h in 1:M) {
    for (k in 1:M) {
      for (i in 1:M) {
        for (j in 1:M) {
          pi <- McCullagh_compute_pi(psi, delta, alpha, c)
          deriv = McCullagh_derivative_pi_wrt_psi(h, k, i, j, psi, delta, alpha, c)

          psi[i, j] <- psi[i, j] - delta1
          psi[j, i] <- psi[i, j]
          pi0 <- McCullagh_compute_pi(psi, delta, alpha, c)

          psi[i, j] <- psi[i, j] +  2.0 * delta1
          psi[j, i] <- psi[i, j]
          pi1 <- McCullagh_compute_pi(psi, delta, alpha, c)

          psi[i, j] <- psi[i, j] - delta1
          psi[j, i] = psi[i, j]

          der = ((pi[h, k] - pi0[h, k]) / delta1 + (pi1[h, k] - pi[h, k]) / delta1) / 2.0
          expect_true(abs(der - deriv) <= delta1,
                      info=paste(h, k, "psi", i, j, der, deriv, sep=" "))
        }
      }
    }
  }
}
)


test_that("McCullagh_derivative_pi_wrt_delta", {
  M <- nrow(psi)
  c <- c_value

  pi <- McCullagh_compute_pi(psi, delta, alpha, c)

  delta <- delta - delta1
  pi0 <- McCullagh_compute_pi(psi, delta, alpha, c)

  delta <- delta + 2.0 * delta1
  pi1 <- McCullagh_compute_pi(psi, delta, alpha, c)

  delta <- delta - delta1
  for (h in 1:M) {
    for (k in 1:M) {
      der = ((pi[h, k] - pi0[h, k]) / delta1 + (pi1[h, k] - pi[h, k]) / delta1) / 2.0
      deriv = McCullagh_derivative_pi_wrt_delta(h, k, psi, delta, alpha)
      expect_true(abs(der - deriv) <= delta1,
                  info=paste("delta ", h, " ", k, " ", der, " ", deriv))
    }
  }
}
)


test_that("McCullagh_derivative_pi_wrt_alpha", {
  M <- nrow(psi)
  c <- c_value

  pi <- McCullagh_compute_pi(psi, delta, alpha, c)

  for (i in 1:M) {
    for (j in 1:M) {
      for (index in 2:(M - 1)) {
        alpha[index] <- alpha[index] - delta1
        pi0 <- McCullagh_compute_pi(psi, delta, alpha, c)

        alpha[index] <- alpha[index] + 2.0 * delta1
        pi1 <- McCullagh_compute_pi(psi, delta, alpha, c)

        alpha[index] <- alpha[index] - delta1

        der <- ((pi[i, j] - pi0[i, j]) / delta1 + (pi1[i, j] - pi[i, j]) / delta1) / 2.0
        deriv <- McCullagh_derivative_pi_wrt_alpha(i, j, index, psi, delta, alpha)
        expect_true(abs(der - deriv) <= delta1, info=paste(i, j, der, deriv))
      }
    }
  }
}
)


test_that("McCullagh_derivative_pi_wrt_c", {
  M <- nrow(psi)
  c <- c_value

  # i and j index pi
  for (i in 1:M) {
    for (j in 1:M) {
      pi <- McCullagh_compute_pi(psi, delta, alpha, c)
      f <- pi[i, j]

      c <- c -delta1
      pi <- McCullagh_compute_pi(psi, delta, alpha, c)
      f0 <- pi[i, j]

      c <- c + 2.0 * delta1
      pi <- McCullagh_compute_pi(psi, delta, alpha, c)
      f1 <- pi[i, j]

      c <- c - delta1

      der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
      deriv <- McCullagh_derivative_pi_wrt_c(i, j, psi, delta, alpha, c)
      expect_true(abs(der - deriv) <= epsilon, info=paste(i, j, der, deriv))
    }
  }
}
)


test_that("McCullagh_derivative_pi_wrt_delta_vec", {
  M <- nrow(psi)
  c <- c_value
  pi <- McCullagh_compute_generalized_pi(psi, delta_vec, alpha, c)

  for (i in 1:M) {
    for (j in 1:M) {
      for (k in 1:(M - 1)) {
        delta_vec[k] <- delta_vec[k] - delta1
        pi0 <- McCullagh_compute_generalized_pi(psi, delta_vec, alpha, c)

        delta_vec[k] <- delta_vec[k] + 2.0 * delta1
        pi1 <- McCullagh_compute_generalized_pi(psi, delta_vec, alpha, c)

        delta_vec[k] <- delta_vec[k] - delta1

        der <- ((pi[i, j] - pi0[i, j]) / delta1 + (pi1[i, j] - pi[i, j]) / delta1) / 2.0
        deriv <- McCullagh_derivative_pi_wrt_delta_vec(i, j, k, psi, delta_vec, alpha, c)
        expect_true(abs(der - deriv) <= delta1, info=paste(i, j, der, deriv))
      }
    }
  }
}
)


test_that("McCullagh_second_order_pi_wrt_psi_2", {
  M <- nrow(psi)
  c <- c_value

  # i1 and j1 index first psi
  for (i1 in 1:M) {
    for (j1 in 1:M) {
      # i2 and j2 index second psi
      for (i2 in 1:M) {
        for (j2 in 1:M) {
          # a and b index pi
          for (a in 1:M) {
            for (b in 1:M) {
              f <- McCullagh_derivative_pi_wrt_psi(a, b, i2, j2, psi, delta, alpha, c)

              psi[i1, j1] <- psi[i1, j1] - delta1
              psi[j1, i1] <- psi[i1, j1]
              f0 <- McCullagh_derivative_pi_wrt_psi(a, b, i2, j2, psi, delta, alpha, c)

              psi[i1, j1] <- psi[i1, j1] +  2.0 * delta1
              psi[j1, i1] <- psi[i1, j1]
              f1 = McCullagh_derivative_pi_wrt_psi(a, b, i2, j2, psi, delta, alpha, c)

              psi[i1, j1] <- psi[i1, j1] - delta1
              psi[j1, i1] <- psi[i1, j1]

              der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
              deriv <- McCullagh_second_order_pi_wrt_psi_2(a, b, i1, j1, i2, j2, psi, delta, alpha, c)
              expect_true(abs(der - deriv) <= epsilon, info=paste(a, b, der, deriv))
            }
          }
        }
      }
    }
  }
}
)


test_that("McCullagh_second_order_pi_wrt_psi_delta", {
  M <- nrow(psi)
  c <- c_value

  # i1 and j1 index psi
  for (i1 in 1:M) {
    for (j1 in 1:M) {
      # a and b index pi
      for (a in 1:M) {
        for (b in 1:M) {
          f <- McCullagh_derivative_pi_wrt_psi(a, b, i1, j1, psi, delta, alpha, c)

          delta <- delta - delta1
          f0 <- McCullagh_derivative_pi_wrt_psi(a, b, i1, j1, psi, delta, alpha, c)

          delta <- delta + 2.0 * delta1
          f1 <- McCullagh_derivative_pi_wrt_psi(a, b, i1, j1, psi, delta, alpha, c)

          delta <- delta - delta1

          der = ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
          deriv <- McCullagh_second_order_pi_wrt_psi_delta(a, b, i1, j1, psi, delta, alpha, c)
          expect_true(abs(der - deriv) <= epsilon, info=paste(a, b, i1, j1, der, deriv))
        }
      }
    }
  }
}
)


test_that("McCullagh_second_order_pi_wrt_psi_alpha", {
  M <- nrow(psi)
  c <- c_value

  # i1 and j1 index psi
  for (i1 in 1:M) {
    for (j1 in i1: M) {
      # index indexes alpha
      for (index in 2:(M - 1)) {
        # i and j index pi
        for (i in 1:M) {
          for (j in 1:M) {
            f <- McCullagh_derivative_pi_wrt_psi(i, j, i1, j1, psi, delta, alpha, c)
            alpha[index] <- alpha[index] - delta1
            f0 <- McCullagh_derivative_pi_wrt_psi(i, j, i1, j1, psi, delta, alpha, c)

            alpha[index] <- alpha[index] + 2.0 * delta1
            f1 <- McCullagh_derivative_pi_wrt_psi(i, j, i1, j1, psi, delta, alpha, c)

            alpha[index] <- alpha[index] - delta1
            der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
            deriv <- McCullagh_second_order_pi_wrt_psi_alpha(i, j, i1, j1, index, psi, delta, alpha, c)
            expect_true(abs(der - deriv) <= epsilon, info=paste(i, j, i1, j1, index, der, deriv))
          }
        }
      }
    }
  }
}
)


test_that("McCullagh_second_order_pi_wrt_psi_c", {
  M <- nrow(psi)
  c <- c_value

  # i1 and j1 index psi
  for (i1 in 1:M) {
    for (j1 in 1:M) {
      # i and j index pi
      for (i in 1:M) {
        for (j in 1:M) {
          f <- McCullagh_derivative_pi_wrt_psi(i, j, i1, j1, psi, delta, alpha, c)

          c <- c - delta1
          f0 <- McCullagh_derivative_pi_wrt_psi(i, j, i1, j1, psi, delta, alpha, c)

          c <- c + 2.0 * delta1
          f1 <- McCullagh_derivative_pi_wrt_psi(i, j, i1, j1, psi, delta, alpha, c)

          c <- c - delta1
          der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
          deriv <- McCullagh_second_order_pi_wrt_psi_c(i, j, i1, j1, psi, delta, alpha, c)
          expect_true(abs(der - deriv) <= epsilon, info=paste("d pi d psi d c", i, j, der, deriv))
        }
      }
    }
  }
}
)


test_that("McCullagh_second_order_pi_wrt_delta_2", {
  M <- nrow(psi)
  c <- c_value

  # i and j index pi
  for (i in 1:M) {
    for (j in 1:M) {
      f <- McCullagh_derivative_pi_wrt_delta(i, j, psi, delta, alpha, c)

      delta <- delta - delta1
      f0 <- McCullagh_derivative_pi_wrt_delta(i, j, psi, delta, alpha, c)

      delta <- delta + 2.0 * delta1
      f1 <- McCullagh_derivative_pi_wrt_delta(i, j, psi, delta, alpha, c)

      delta <- delta - delta1

      der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
      deriv <- McCullagh_second_order_pi_wrt_delta_2(i, j, psi, delta, alpha, c)
      expect_true(abs(der - deriv) <= epsilon, info=paste(i, j, der, deriv))
    }
  }
}
)


test_that("McCullagh_second_order_pi_wrt_delta_alpha", {
  M <- nrow(psi)
  c <- c_value

  # i and j index pi
  for (i in 1:M) {
    for (j in 1:M) {
      # index indexes alpha
      for (index in 2:(M - 1)) {
        f <- McCullagh_derivative_pi_wrt_alpha(i, j, index, psi, delta, alpha, c)

        delta <- delta - delta1
        f0 <- McCullagh_derivative_pi_wrt_alpha(i, j, index, psi, delta, alpha, c)

        delta <- delta + 2.0 * delta1
        f1 <- McCullagh_derivative_pi_wrt_alpha(i, j, index, psi, delta, alpha, c)

        delta <- delta - delta1

        der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
        deriv <- McCullagh_second_order_pi_wrt_delta_alpha(i, j, index, psi, delta, alpha, c)
        expect_true(abs(der - deriv) <= epsilon, info=paste(a, b, i, j, index, der, deriv))
      }
    }
  }
}
)


test_that("McCullagh_second_order_pi_wrt_delta_c", {
  M <- nrow(psi)
  c <- c_value

  # i and j index pi
  for (i in 1:M) {
    for (j in 1:M) {
      f = McCullagh_derivative_pi_wrt_delta(i, j, psi, delta, alpha, c)

      c < c - delta1
      f0 <- McCullagh_derivative_pi_wrt_delta(i, j, psi, delta, alpha, c)

      c <- c + 2.0 * delta1
      f1 <- McCullagh_derivative_pi_wrt_delta(i, j, psi, delta, alpha, c)

      c <- c - delta1

      der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
      deriv <- McCullagh_second_order_pi_wrt_delta_c(i, j, psi, delta, alpha, c)
      expect_true(abs(der - deriv) <= epsilon, info=paste(i, j, der, deriv))
    }
  }
}
)


test_that("McCullagh_second_order_pi_wrt_alpha_2", {
  M <- nrow(psi)
  c <- c_value

  # i and j index pi
  for (i in 1:M) {
    for (j in 1:M) {
      # index1 and index2 index alpha
      for (index1 in 2:(M - 1)) {
        for (index2 in 2:(M - 1)) {
          f <- McCullagh_derivative_pi_wrt_alpha(i, j, index1, psi, delta, alpha, c)

          alpha[index2] <- alpha[index2] - delta1
          f0 <- McCullagh_derivative_pi_wrt_alpha(i, j, index1, psi, delta, alpha, c)

          alpha[index2] <- alpha[index2] + 2.0 * delta1
          f1 <- McCullagh_derivative_pi_wrt_alpha(i, j, index1, psi, delta, alpha, c)

          alpha[index2] <- alpha[index2] - delta1

          der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
          deriv <- McCullagh_second_order_pi_wrt_alpha_2(i, j, index1, index2, psi, delta, alpha, c)
          expect_true(abs(der - deriv) <= epsilon, info=paste("alpha^2", i, j, index1, index2, der, deriv))
        }
      }
    }
  }
}
)


test_that("McCullagh_second_order_pi_wrt_alpha_c", {
  M <- nrow(psi)
  c <- c_value

  # in and j index pi
  for (i in 1:M) {
    for (j in 1:M) {
      # index indexes alpha
      for (index in 2:(M - 1)) {
        f <- McCullagh_derivative_pi_wrt_alpha(i, j, index, psi, delta, alpha, c)

        c <- c - delta1
        f0 <- McCullagh_derivative_pi_wrt_alpha(i, j, index, psi, delta, alpha, c)

        c <- c + 2.0 * delta1
        f1 <- McCullagh_derivative_pi_wrt_alpha(i, j, index, psi, delta, alpha, c)

        c <- c - delta1

        der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
        deriv <- McCullagh_second_order_pi_wrt_alpha_c(i, j, index, psi, delta, alpha, c)
        expect_true(abs(der - deriv) <= epsilon, info=paste(i, j, index, der, deriv))
      }
    }
  }
}
)


test_that("McCullagh_second_order_pi_wrt_c_2", {
  M <- nrow(psi)
  c <- c_value

  # i and j index pi
  for (i in 1:M) {
    for (j in 1:M) {
      f <- McCullagh_derivative_pi_wrt_c(i, j, psi, delta, alpha, c)

      c <- c - delta1
      f0 <- McCullagh_derivative_pi_wrt_c(i, j, psi, delta, alpha, c)

      c <- c + 2.0 * delta1
      f1 <- McCullagh_derivative_pi_wrt_c(i, j, psi, delta, alpha, c)

      c <- c - delta1

      der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
      deriv <- McCullagh_second_order_pi_wrt_c_2(i, j, psi, delta, alpha, c)
      expect_true(abs(der - deriv) <= epsilon, info=paste(i, j, der, deriv))
    }
  }
}
)


test_that("McCullagh_second_order_pi_wrt_psi_delta_vec", {
  M <- nrow(psi)
  c <- c_value

  for (k in 1:(M - 1)) {
    # i1 and j1 index psi
    for (i1 in 1:(M)) {
      for (j1 in 1:(M)) {
        # i and j index pi
        for (i in 1:M) {
          for (j in 1:M) {
            f <- McCullagh_derivative_pi_wrt_delta_vec(i, j, k, psi, delta_vec, alpha, c)

            psi[i1, j1] <- psi[i1, j1] - delta1
            psi[j1, i1] <- psi[i1, j1]
            f0 <- McCullagh_derivative_pi_wrt_delta_vec(i, j, k, psi, delta_vec, alpha, c)

            psi[i1, j1] <- psi[i1, j1] + 2.0 * delta1
            psi[j1, i1] <- psi[i1, j1]
            f1 <- McCullagh_derivative_pi_wrt_delta_vec(i, j, k, psi, delta_vec, alpha, c)

            psi[i1, j1] <- psi[i1, j1] - delta1
            psi[j1, i1] <- psi[i1, j1]

            der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
            deriv <- McCullagh_second_order_pi_wrt_psi_delta_vec(i, j, i1, j1, k, psi, delta_vec, alpha, c)
            expect_true(abs(der - deriv) <= epsilon,
                        info=paste("d psi d delta_vec", k, i1, j1, i, j, der, deriv))
          }
        }
      }
    }
  }
}
)


test_that("McCullagh_second_order_pi_wrt_delta_vec_2", {
  M <- nrow(psi)
  c <- c_value

  for (k1 in 1:(M - 1)) {
    for (k2 in 1:(M - 1)) {
      # i and j index pi
      for (a in 1:M) {
        for (b in 1:M) {
          f <- McCullagh_derivative_pi_wrt_delta_vec(a, b, k2, psi, delta_vec, alpha, c)

          delta_vec[k1] <- delta_vec[k1] - delta1
          f0 <- McCullagh_derivative_pi_wrt_delta_vec(a, b, k2, psi, delta_vec, alpha, c)

          delta_vec[k1] <- delta_vec[k1] + 2.0 * delta1
          f1 <- McCullagh_derivative_pi_wrt_delta_vec(a, b, k2, psi, delta_vec, alpha, c)

          delta_vec[k1] <- delta_vec[k1] - delta1

          der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
          deriv <- McCullagh_second_order_pi_wrt_delta_vec_2(a, b, k1, k2, psi, delta_vec, alpha, c)
          expect_true(abs(der - deriv) <= epsilon, info=paste(k1, k2, i, j, der, deriv))
        }
      }
    }
  }
}
)


test_that("McCullagh_second_order_pi_wrt_delta_vec_alpha", {
  M <- nrow(psi)
  c <- c_value

  for (k in 1:(M - 1)) {
    for (index in 2:(M - 1)) {
      for (i in 1:M) {
        for (j in 1:M) {
          f <- McCullagh_derivative_pi_wrt_delta_vec(i, j, k, psi, delta_vec, alpha, c)

          alpha[index] <- alpha[index] - delta1
          f0 <- McCullagh_derivative_pi_wrt_delta_vec(i, j, k, psi, delta_vec, alpha, c)

          alpha[index] <- alpha[index] + 2.0 * delta1
          f1 <- McCullagh_derivative_pi_wrt_delta_vec(i, j, k, psi, delta_vec, alpha, c)

          alpha[index] <- alpha[index] - delta1

          der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
          deriv <- McCullagh_second_order_pi_wrt_delta_vec_alpha(i, j, k, index, psi, delta_vec, alpha, c)
          expect_true(abs(der - deriv) <= epsilon,
                      info=paste("d delta_vec d alpha", k, index, i, j, der, deriv))
        }
      }
    }
  }
}
)


test_that("McCullagh_second_order_pi_wrt_delta_vec_c", {
  M <- nrow(psi)
  c <- c_value

  for (k in 1:(M - 1)) {
    for (i in 1:M) {
      for (j in 1:M) {
        f <- McCullagh_derivative_pi_wrt_delta_vec(i, j, k, psi, delta_vec, alpha, c)

        c <- c - delta1
        f0 <- McCullagh_derivative_pi_wrt_delta_vec(i, j, k, psi, delta_vec, alpha, c)

        c <- c + 2.0 * delta1
        f1 <- McCullagh_derivative_pi_wrt_delta_vec(i, j, k, psi, delta_vec, alpha, c)

        c <- c - delta1

        der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
        deriv <- McCullagh_second_order_pi_wrt_delta_vec_c(i, j, k, psi, delta_vec, alpha, c)
        expect_true(abs(der - deriv) <= epsilon,
                    info=paste("d delta_vec d alpha", k, index, i, j, der, deriv))
      }
    }
  }
}
)


test_that("McCullagh_derivative_condition_wrt_psi", {
  M <- nrow(psi)
  c <- c_value

  for (i1 in 1:M) {
    for (j1 in i1:M) {
      f <- McCullagh_compute_condition(psi)

      psi[i1, j1] <- psi[i1, j1] - delta1
      psi[j1, i1] <- psi[i1, j1]
      f0 <- McCullagh_compute_condition(psi)

      psi[i1, j1] <- psi[i1, j1] + 2.0 * delta1
      psi[j1, i1] <- psi[i1, j1]
      f1 <- McCullagh_compute_condition(psi)

      psi[i1, j1] <- psi[i1, j1] - delta1
      psi[j1, i1] <- psi[i1, j1]

      der = ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
      deriv = McCullagh_derivative_condition_wrt_psi(i1, j1)
      expect_true(abs(der - deriv) <= epsilon,
                  info=paste("condition", i1, j1, der, deriv))
    }
  }
}
)


test_that("McCullagh_deriv_omega_wrt_psi", {
  M <- nrow(psi)
  c <- c_value

  pi <- McCullagh_compute_pi(psi, delta, alpha, c)
  f <- McCullagh_compute_omega(n, pi)

  for (i1 in 1:M) {
    for (j1 in i1:M) {
      deriv <- McCullagh_derivative_omega_wrt_psi(n, i1, j1, psi, delta, alpha, c)

      psi[i1, j1] <- psi[i1, j1] - delta1
      psi[j1, i1] <- psi[i1, j1]
      pi0 <- McCullagh_compute_pi(psi, delta, alpha, c)
      f0 <- McCullagh_compute_omega(n, pi0)

      psi[i1, j1] <- psi[i1, j1] + 2.0 * delta1
      psi[j1, i1] <- psi[i1, j1]
      pi1 <- McCullagh_compute_pi(psi, delta, alpha, c)
      f1 <- McCullagh_compute_omega(n, pi1)

      psi[i1, j1] <- delta1
      psi[j1, i1] <- psi[i1, j1]

      der = ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
      expect_true(abs(der - deriv) <= epsilon, info=paste("omega-alpha", i1, j1, der, deriv))
    }
  }
}
)


test_that("McCullagh_deriv_omega_wrt_delta", {
  M <- nrow(psi)
  c <- c_value

  pi <- McCullagh_compute_pi(psi, delta, alpha, c)
  f <- McCullagh_compute_omega(n, pi)
  deriv <- McCullagh_derivative_omega_wrt_delta(n, psi, delta, alpha, c)

  delta <- delta - delta1
  pi0 <- McCullagh_compute_pi(psi, delta, alpha, c)
  f0 <- McCullagh_compute_omega(n, pi0)

  delta <- delta + 2.0 * delta1
  pi1 <- McCullagh_compute_pi(psi, delta, alpha, c)
  f1 <- McCullagh_compute_omega(n, pi1)

  delta <- delta - delta1

  der = ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
  expect_true(abs(der - deriv) <= epsilon, info=paste("omega-delta", der, deriv))
}
)


test_that("McCullagh_deriv_omega_wrt_alpha", {
  M <- nrow(psi)
  c <- c_value

  pi <- McCullagh_compute_pi(psi, delta, alpha, c)

  for (index in 2:(M - 1)) {
    f <- McCullagh_compute_omega(n, pi)
    deriv <- McCullagh_derivative_omega_wrt_alpha(n, index, psi, delta, alpha, c)

    alpha[index] <- alpha[index] - delta1
    pi0 <- McCullagh_compute_pi(psi, delta, alpha, c)
    f0 <- McCullagh_compute_omega(n, pi0)

    alpha[index] <- alpha[index] + 2.0 * delta1
    pi1 <- McCullagh_compute_pi(psi, delta, alpha, c)
    f1 <- McCullagh_compute_omega(n, pi1)

    alpha[index] <- alpha[index] - delta1

    der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
    if (deriv != 0.0) {
      expect_true(abs((der - deriv) / deriv) <= epsilon, info=paste("d omega / d alpha", index, der, deriv))
    } else {
      expect_true(abs(der - deriv) <= epsilon, info=paste("d omega / d alpha", index, der, deriv))
    }
  }
}
)


test_that("McCullagh_deriv_omega_wrt_c", {
  M <- nrow(psi)
  c <- c_value

  pi <- McCullagh_compute_pi(psi, delta, alpha, c)
  f <- McCullagh_compute_omega(n, pi)

  c <- c - delta1
  pi0 <- McCullagh_compute_pi(psi, delta, alpha, c)
  f0 <- McCullagh_compute_omega(n, pi0)

  c <- c + 2.0 * delta1
  pi1 <- McCullagh_compute_pi(psi, delta, alpha, c)
  f1 <- McCullagh_compute_omega(n, pi1)

  c <- c - delta1

  der = ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
  deriv <- McCullagh_derivative_omega_wrt_c(n, psi, delta, alpha, c)
  expect_true(abs(der - deriv) <= epsilon, info=paste("omega-c", der, deriv))
}
)


test_that("McCullagh_deriv_omega_wrt_delta_vec", {
  M <- nrow(psi)
  c <- c_value

  pi <- McCullagh_compute_generalized_pi(psi, delta_vec, alpha, c)

  f <- McCullagh_compute_omega(n, pi)
  for (k in 1:(M-1)) {
    delta_vec[k] <- delta_vec[k] - delta1
    pi0 <- McCullagh_compute_generalized_pi(psi, delta_vec, alpha, c)
    f0 <- McCullagh_compute_omega(n, pi0)

    delta_vec[k] <- delta_vec[k] + 2.0 * delta1
    pi1 <- McCullagh_compute_generalized_pi(psi, delta_vec, alpha, c)
    f1 <- McCullagh_compute_omega(n, pi1)

    delta_vec[k] <- delta_vec[k] - delta1

    der = ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
    deriv <- McCullagh_derivative_omega_wrt_delta_vec(n, k, psi, delta_vec, alpha, c)
    expect_true(abs(der - deriv) <= epsilon, info=paste("omega-delta", der, deriv))
  }
}
)


test_that("McCullagh_second_order_omega_wrt_psi_2", {
  M <- nrow(psi)
  c <- c_value

  f <- McCullagh_derivative_omega_wrt_delta(n, psi, delta, alpha, c)
  for (i1 in 1:M) {
    for (j1 in i1:M) {
      for (i2 in 1:M) {
        for (j2 in i2:M) {
          psi[i1, j1] <- psi[i1, j1] - delta1
          psi[j1, i1] <- psi[i1, j1]
          f0 <- McCullagh_derivative_omega_wrt_psi(n, i2, j2, psi, delta, alpha, c)

          psi[i1, j1] <- psi[i1, j1] + 2.0 * delta1
          psi[j1, i1] <- psi[i1, j1]
          f1 <- McCullagh_derivative_omega_wrt_psi(n, i2, j2, psi, delta, alpha, c)

          psi[i1, j1] <- psi[i1, j1] - delta1
          psi[j1, i1] = psi[i1, j1]
          der = ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
          deriv <- McCullagh_second_order_omega_wrt_psi_2(n, i1, j1, i2, j2, psi, delta, alpha, c)
          expect_true(abs(der - deriv) <= epsilon, info=paste("omega d psi^2", i1, j1, i2, j2, der, deriv))
        }
      }
    }
  }
}
)


test_that("McCullagh_second_order_omega_wrt_psi_delta", {
  M <- nrow(psi)
  c <- c_value

  f <- McCullagh_derivative_omega_wrt_delta(n, psi, delta, alpha, c)
  for (i1 in 1:M) {
    for (j1 in i1:M) {
      deriv <- McCullagh_second_order_omega_wrt_psi_delta(n, i1, j1, psi, delta, alpha, c)

      psi[i1, j1] <- psi[i1, j1] - delta1
      psi[j1, i1] <- psi[i1, j1]
      f0 <- McCullagh_derivative_omega_wrt_delta(n, psi, delta, alpha, c)

      psi[i1, j1] <- psi[i1, j1] + 2.0 * delta1
      psi[j1, i1] <- psi[i1, j1]
      f1 <- McCullagh_derivative_omega_wrt_delta(n, psi, delta, alpha, c)

      psi[i1, j1] <- psi[i1, j1] - delta1
      psi[j1, i1] <- psi[i1, j1]
      der = ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
      expect_true(abs(der - deriv) <= epsilon, info=paste(i1, j1, der, deriv))
    }
  }
}
)


test_that("McCullagh_second_order_omega_wrt_psi_alpha", {
  M <- nrow(psi)
  c <- c_value

  for (i1 in 1:M) {
    for (j1 in i1:M) {
      for (index in 2:(M - 1)) {
        f <- McCullagh_derivative_omega_wrt_alpha(n, index, psi, delta, alpha, c)

        psi[i1, j1] <- psi[i1, j1] - delta1
        f0 <- McCullagh_derivative_omega_wrt_alpha(n, index, psi, delta, alpha, c)

        psi[i1, j1] <- psi[i1, j1] + 2.0 * delta1
        f1 <- McCullagh_derivative_omega_wrt_alpha(n, index, psi, delta, alpha, c)

        psi[i1, j1] <- psi[i1, j1] - delta1

        der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
        deriv <- McCullagh_second_order_omega_wrt_psi_alpha(n, i1, j1, index, psi, delta, alpha, c)
        expect_true(abs(der - deriv) <= epsilon, info=paste("d^2 omega / d psi d alpha", i, j, der, deriv))
      }
    }
  }
}
)


test_that("McCullagh_second_order_omega_wrt_psi_c", {
  M <- nrow(psi)
  c <- c_value

  for (i1 in 1:M) {
    for (j1 in 1:M) {
      f <- McCullagh_derivative_omega_wrt_c(n, psi, delta, alpha, c)

      psi[i1, j1] <- psi[i1, j1] - delta1
      psi[j1, i1] <- psi[i1, j1]
      f0 <- McCullagh_derivative_omega_wrt_c(n, psi, delta, alpha, c)

      psi[i1, j1] <- psi[i1, j1] + 2.0 * delta1
      psi[j1, i1] <- psi[i1, j1]
      f1 <- McCullagh_derivative_omega_wrt_c(n, psi, delta, alpha, c)

      psi[i1, j1] <- psi[i1, j1] - delta1
      psi[j1, i1] <- psi[i1, j1]

      der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
      deriv <- McCullagh_second_order_omega_wrt_psi_c(n, i1, j1, psi, delta, alpha, c)
      expect_true(abs(der - deriv) <= epsilon, info=paste("d^2 omega d psi d c", i1, j1, der, deriv))
    }
  }
}
)


test_that("McCullagh_second_order_omega_wrt_delta_2", {
  c <- c_value

  f <- McCullagh_derivative_omega_wrt_delta(n, psi, delta, alpha, c)

  delta <- delta - delta1
  f0 <- McCullagh_derivative_omega_wrt_delta(n, psi, delta, alpha, c)

  delta <- delta + 2.0 * delta1
  f1 <- McCullagh_derivative_omega_wrt_delta(n, psi, delta, alpha, c)

  delta <- delta - delta1

  der = ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
  deriv = McCullagh_second_order_omega_wrt_delta_2(n, psi, delta, alpha, c)
  expect_true(abs(der - deriv) <= epsilon, info=paste("d^2 omega d delta^2", der, deriv))
}
)


test_that("McCullagh_second_order_omega_wrt_delta_alpha", {
  M <- nrow(psi)
  c <- c_value

  for (index in 2:(M - 1)) {
    f <- McCullagh_derivative_omega_wrt_delta(n, psi, delta, alpha, c)

    alpha[index] <- alpha[index] - delta1
    f0 <- McCullagh_derivative_omega_wrt_delta(n, psi, delta, alpha, c)

    alpha[index] <- alpha[index] + 2.0 * delta1
    f1 <- McCullagh_derivative_omega_wrt_delta(n, psi, delta, alpha, c)

    alpha[index] <- alpha[index] - delta1

    der = ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
    deriv = McCullagh_second_order_omega_wrt_delta_alpha(n, index, psi, delta, alpha, c)
    expect_true(abs(der - deriv) <= epsilon, info=paste("d^2 omega d delta d alpha", index, der, deriv))
  }
}
)


test_that("McCullagh_second_order_omega_wrt_delta_c", {
  c <- c_value

  f <- McCullagh_derivative_omega_wrt_c(n, psi, delta, alpha, c)

  delta <- delta - delta1
  f0 <- McCullagh_derivative_omega_wrt_c(n, psi, delta, alpha, c)

  delta <- delta + 2.0 * delta1
  f1 <- McCullagh_derivative_omega_wrt_c(n, psi, delta, alpha, c)

  delta <- delta - delta1

  der = ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
  deriv = McCullagh_second_order_omega_wrt_delta_c(n, psi, delta, alpha, c)
  expect_true(abs(der - deriv) <= epsilon, info=paste("d^2 omega d delta d c", der, deriv))
}
)


test_that("McCullagh_second_order_omega_wrt_alpha_2", {
  M <- nrow(psi)
  c <- c_value

  for (i in 2:(M - 1)) {
    for (j in 2:(M - 1)) {
      f <- McCullagh_derivative_omega_wrt_alpha(n, j, psi, delta, alpha, c)

      alpha[i] <- alpha[i] - delta1
      f0 <- McCullagh_derivative_omega_wrt_alpha(n, j, psi, delta, alpha, c)

      alpha[i] <- alpha[i] + 2.0 * delta1
      f1 <- McCullagh_derivative_omega_wrt_alpha(n, j, psi, delta, alpha, c)

      alpha[i] <- alpha[i] - delta1

      der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
      deriv <- McCullagh_second_order_omega_wrt_alpha_2(n, i, j, psi, delta, alpha, c)
      expect_true(abs(der - deriv) <= epsilon, info=paste("d^2 omega / d alpha^2", i, j, der, deriv))
    }
  }
}
)


test_that("McCullagh_second_order_omega_wrt_alpha_c", {
  M <- nrow(psi)
  c <- c_value

  for (index in 2:(M - 1)) {
    f <- McCullagh_derivative_omega_wrt_c(n, psi, delta, alpha, c)

    alpha[index] <- alpha[index] - delta1
    f0 <- McCullagh_derivative_omega_wrt_c(n, psi, delta, alpha, c)

    alpha[index] <- alpha[index] + 2.0 * delta1
    f1 <- McCullagh_derivative_omega_wrt_c(n, psi, delta, alpha, c)

    alpha[index] <- alpha[index] - delta1

    der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
    deriv <- McCullagh_second_order_omega_wrt_alpha_c(n, index, psi, delta, alpha, c)
    if (deriv != 0.0) {
      expect_true(abs((der - deriv) / deriv) <= epsilon, info=paste("d^2 omega d alpha d c", index, der, deriv))

    } else {
      expect_true(abs(der - deriv) <= epsilon, info=paste("d^2 omega d alpha d c", index, der, deriv))
    }
  }
}
)


test_that("McCullagh_second_order_omega_wrt_c_2", {
  c <- c_value

  f <- McCullagh_derivative_omega_wrt_c(n, psi, delta, alpha, c)

  c <- c - delta1
  f0 <- McCullagh_derivative_omega_wrt_c(n, psi, delta, alpha, c)

  c <- c + 2.0 * delta1
  f1 = McCullagh_derivative_omega_wrt_c(n, psi, delta, alpha, c)

  c <- delta1

  der = ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
  deriv = McCullagh_second_order_omega_wrt_c_2(n, psi, delta, alpha, c)
  expect_true(abs(der - deriv) <= epsilon, info=paste("d^2 omega d c^2", der,deriv))
}
)


test_that("McCullagh_second_order_omega_wrt_psi_delta_vec", {
  M <- nrow(psi)
  c <- c_value

  for (k in 1:(M - 1)) {
    f <- McCullagh_derivative_omega_wrt_delta_vec(n, k, psi, delta_vec, alpha, c)

    for (i1 in 1:M) {
      for (j1 in i1:M) {
        psi[i1, j1] <- psi[i1, j1] - delta1
        psi[j1, i1] <- psi[i1, j1]
        f0 <- McCullagh_derivative_omega_wrt_delta_vec(n, k, psi, delta_vec, alpha, c)

        psi[i1, j1] <- psi[i1, j1] + 2.0 * delta1
        psi[j1, i1] <- psi[i1, j1]
        f1 <- McCullagh_derivative_omega_wrt_delta_vec(n, k, psi, delta_vec, alpha, c)

        psi[i1, j1] <- psi[i1, j1] - delta1
        psi[j1, i1] <- psi[i1, j1]

        der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
        deriv <- McCullagh_second_order_omega_wrt_psi_delta_vec(n, i1, j1, k, psi, delta_vec, alpha, c)
        expect_true(abs(der - deriv) <= epsilon,
                    info=paste("d^2 omega / d psi d delta_vec", i1, j1, der, deriv))
      }
    }
  }
}
)


test_that("McCullagh_second_order_omega_wrt_delta_vec_2", {
  M <- nrow(psi)
  c <- c_value

  for (k1 in 1:(M - 1)) {
    for (k2 in 1:(M - 1)) {
      f <- McCullagh_derivative_omega_wrt_delta_vec(n, k2, psi, delta_vec, alpha, c)

      delta_vec[k1] <- delta_vec[k1] - delta1
      f0 <- McCullagh_derivative_omega_wrt_delta_vec(n, k2, psi, delta_vec, alpha, c)

      delta_vec[k1] <- delta_vec[k1] + 2.0 * delta1
      f1 <- McCullagh_derivative_omega_wrt_delta_vec(n, k2, psi, delta_vec, alpha, c)

      delta_vec[k1] <- delta_vec[k1] - delta1

      der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
      deriv <- McCullagh_second_order_omega_wrt_delta_vec_2(n, k1, k2, psi, delta_vec, alpha, c)
      expect_true(abs(der - deriv) <= epsilon, info=paste("d^2 omega d delta_vec^2", k1, k2, der, deriv))
    }
  }
}
)


test_that("McCullagh_second_order_omega_wrt_delta_vec_alpha", {
  M <- nrow(psi)
  c <- c_value

  for (k in 1:(M - 1)) {
    for (index in 2:(M - 1)) {
      f <- McCullagh_derivative_omega_wrt_delta_vec(n, k, psi, delta_vec, alpha, c)

      alpha[index] <- alpha[index] - delta1
      f0 <- McCullagh_derivative_omega_wrt_delta_vec(n, k, psi, delta_vec, alpha, c)

      alpha[index] <- alpha[index] + 2.0 * delta1
      f1 <- McCullagh_derivative_omega_wrt_delta_vec(n, k, psi, delta_vec, alpha, c)

      alpha[index] <- alpha[index] - delta1

      der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
      deriv <- McCullagh_second_order_omega_wrt_delta_vec_alpha(n, k, index, psi, delta_vec, alpha, c)
      expect_true(abs(der - deriv) <= epsilon,
                  info=paste("d^2 omega d delta_vec d alpha", k, index, der, deriv))
    }
  }
}
)


test_that("McCullagh_second_order_omega_wrt_delta_vec_c", {
  M <- nrow(psi)
  c <- c_value

  for (k in 1:(M - 1)) {
    f <- McCullagh_derivative_omega_wrt_delta_vec(n, k, psi, delta_vec, alpha, c)

    c <- c - delta1
    f0 <- McCullagh_derivative_omega_wrt_delta_vec(n, k, psi, delta_vec, alpha, c)

    c <- c + 2.0 * delta1
    f1 <- McCullagh_derivative_omega_wrt_delta_vec(n, k, psi, delta_vec, alpha, c)

    c <- c - delta1

    der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
    deriv <- McCullagh_second_order_omega_wrt_delta_vec_c(n, k, psi, delta_vec, alpha, c)
    expect_true(abs(der - deriv) <= epsilon, info=paste("d^2 omega d delta_vec d c: ", k, der, deriv))
  }
}
)


test_that("McCullagh_derivative_lagrangian_wrt_psi", {
  M <- nrow(psi)
  c <- c_value

  condition <- McCullagh_compute_condition(psi)
  pi <- McCullagh_compute_pi(psi, delta, alpha, c)
  omega <- McCullagh_compute_omega(n, pi)

  f <- omega * condition
  for (i1 in 1:M) {
    for (j1 in i1:M) {
      psi[i1, j1] <- psi[i1, j1] - delta1
      psi[j1, i1] = psi[i1, j1]
      condition0 <- McCullagh_compute_condition(psi)
      pi0 <- McCullagh_compute_pi(psi, delta, alpha, c)
      omega0 <- McCullagh_compute_omega(n, pi0)
      f0 <- omega0 * condition0

      psi[i1, j1] <- psi[i1, j1] + 2.0 * delta1
      psi[j1, i1] <- psi[i1, j1]
      condition1 <- McCullagh_compute_condition(psi)
      pi1 <- McCullagh_compute_pi(psi, delta, alpha, c)
      omega1 <- McCullagh_compute_omega(n, pi1)
      f1 <- omega1 * condition1

      psi[i1, j1] <- psi[i1, j1] - delta1
      psi[j1, i1] <- psi[i1, j1]

      der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
      deriv <- McCullagh_derivative_lagrangian_wrt_psi(n, i1, j1, psi, delta, alpha)
      expect_true(abs(der - deriv) <= epsilon, info=paste("lagrangian psi", i1, j1, der, deriv))
    }
  }
}
)


test_that("McCullagh_second_order_lagrangian_wrt_psi_2", {
  M <- nrow(psi)
  c <- c_value

  for (i1 in 1:M) {
    for (j1 in i1:M) {
      for (i2 in 1:M) {
        for (j2 in i2:M) {
          f <- McCullagh_derivative_lagrangian_wrt_psi(n, i2, j2, psi, delta, alpha, c)

          psi[i1, j1] <- psi[i1, j1] - delta1
          psi[j1, i1] = psi[i1, j1]
          f0 <- McCullagh_derivative_lagrangian_wrt_psi(n, i2, j2, psi, delta, alpha, c)

          psi[i1, j1] <- psi[i1, j1] + 2.0 * delta1
          psi[j1, i1] <- psi[i1, j1]
          f1 <- McCullagh_derivative_lagrangian_wrt_psi(n, i2, j2, psi, delta, alpha, c)

          psi[i1, j1] <- psi[i1, j1] - delta1
          psi[j1, i1] <- psi[i1, j1]

          der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
          deriv <- McCullagh_second_order_lagrangian_wrt_psi_2(n, i1, j1, i2, j2, psi, delta, alpha)
          if (deriv == 0) {
            expect_true(abs(der - deriv) <= epsilon, info=paste("d^2 lagrangian d psi^2", i1, j1, der, deriv))
          } else {
            expect_true(abs((der - deriv) / deriv) <= epsilon, info=paste("d^2 lagrangian d psi^2", i1, j1, der, deriv))
          }
        }
      }
    }
  }
}
)


test_that("McCullagh_second_order_lagrangian_wrt_psi_delta", {
  M <- nrow(psi)
  c <- c_value

  for (i1 in 1:M) {
    for (j1 in i1:M) {
      f <- McCullagh_derivative_lagrangian_wrt_psi(n, i1, j1, psi, delta, alpha, c)

      delta <- delta - delta1
      f0 <- McCullagh_derivative_lagrangian_wrt_psi(n, i1, j1, psi, delta, alpha, c)

      delta <- delta + 2.0 * delta1
      f1 <- McCullagh_derivative_lagrangian_wrt_psi(n, i1, j1, psi, delta, alpha, c)

      delta <- delta - delta1
      der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
      deriv <- McCullagh_second_order_lagrangian_wrt_psi_delta(n, i1, j1, psi, delta, alpha, c)
      if (deriv == 0.0) {
        expect_true(abs(der - deriv) <= epsilon, info=paste("lagrange psi delta", i1, j1, der, deriv))
      } else {
        expect_true(abs((der - deriv) / deriv) <= epsilon, info=paste("lagrange psi delta", i1, j1, der, deriv))
      }
    }
  }
}
)


test_that("McCullagh_second_order_lagrangian_wrt_psi_alpha", {
  M = nrow(psi)
  c <- c_value

  for (i1 in 1:M) {
    for (j1 in i1:M) {
      for (index in 2:(M-1)) {
        f <- McCullagh_derivative_lagrangian_wrt_psi(n, i1, j1, psi, delta, alpha, c)

        alpha[index] <- alpha[index] - delta1
        f0 <- McCullagh_derivative_lagrangian_wrt_psi(n, i1, j1, psi, delta, alpha, c)

        alpha[index] <- alpha[index] + 2.0 * delta1
        f1 <- McCullagh_derivative_lagrangian_wrt_psi(n, i1, j1, psi, delta, alpha, c)

        alpha[index] <- alpha[index] - delta1

        der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
        deriv <- McCullagh_second_order_lagrangian_wrt_psi_alpha(n, i1, j1, index, psi, delta, alpha, c)
        expect_true(abs(der - deriv) <= epsilon, info=paste("lagrange psi alpha", i1, j1, der, deriv))
      }
    }
  }
}
)


test_that("McCullagh_second_order_lagrangian_wrt_psi_delta_vec", {
  M <- nrow(psi)
  c <- c_value

  for (k in 1:(M - 1)) {
    for (i1 in 1:M) {
      for (j1 in i1:M) {
        f = McCullagh_derivative_lagrangian_wrt_delta_vec(n, k, psi, delta_vec, alpha, c)

        psi[i1, j1] <- psi[i1, j1] - delta1
        psi[j1, i1] <- psi[i1, j1]
        f0 <- McCullagh_derivative_lagrangian_wrt_delta_vec(n, k, psi, delta_vec, alpha, c)

        psi[i1, j1] <- psi[i1, j1] + 2.0 * delta1
        psi[j1, i1] <- psi[i1, j1]
        f1 <- McCullagh_derivative_lagrangian_wrt_delta_vec(n, k, psi, delta_vec, alpha, c)

        psi[i1, j1] <- psi[i1, j1] - delta1
        psi[j1, i1] <- psi[i1, j1]

        der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
        deriv <- McCullagh_second_order_lagrangian_wrt_psi_delta_vec(n, i1, j1, k, psi, delta_vec, alpha, c)
        if (deriv == 0.0) {
          expect_true(abs(der - deriv) <= epsilon, info=paste("lagrange psi delta", i1, j1, der, deriv))
        } else {
          expect_true(abs((der - deriv) / deriv) <= epsilon, info=paste("lagrange psi delta", i1, j1, der, deriv))
        }
      }
    }
  }
}
)


test_that("McCullagh_deriv_log_l_wrt_psi", {
  M <- nrow(psi)
  c <- c_value

  for (i in 1:M) {
    for (j in 1:M) {
      f <- McCullagh_log_L(n, psi, delta, alpha)
      derivative <- McCullagh_derivative_log_l_wrt_psi(n, i, j, psi, delta, alpha, c)

      psi[i, j] <- psi[i, j] - delta1
      psi[j, i] <- psi[i, j]
      f0 <- McCullagh_log_L(n, psi, delta, alpha, c)

      psi[i, j] <- psi[i, j] + 2.0 * delta1
      psi[j, i] <- psi[i, j]
      f1 <- McCullagh_log_L(n, psi, delta, alpha, c)

      psi[i, j] <- psi[i, j] - delta1
      psi[j, i] <- psi[i, j]
      der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
      if (derivative != 0.0 && der != 0.0) {
        expect_true(abs((der - derivative) / derivative) <= epsilon * 10.0,
                    info=paste("d logL / d psi", i, j, der, derivative))
      } else {
        expect_true(abs(der - derivative) <= epsilon,
                    info=paste("d logL / d psi", i, j, der, derivative))
      }
    }
  }
}
)


test_that("McCullagh_deriv_log_l_wrt_delta", {
  c <- c_value

  f <- McCullagh_log_L(n, psi, delta, alpha, c)
  derivative <- McCullagh_derivative_log_l_wrt_delta(n, psi, delta, alpha, c)

  delta <- delta - delta1
  f0 <- McCullagh_log_L(n, psi, delta, alpha, c)

  delta <- delta + 2.0 * delta1
  f1 <- McCullagh_log_L(n, psi, delta, alpha, c)

  delta <- delta - delta1
  der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
  if (derivative != 0) {
    expect_true(abs((der - derivative) / derivative) <= epsilon,
                info=paste("der logL wrt delta", der, derivative))
  } else  {
    expect_true(abs(der - derivative) <= epsilon,
              info=paste("der logL wrt delta", der, derivative))
  }
}
)


test_that("McCullagh_deriv_log_l_wrt_alpha", {
  M <- nrow(psi)
  c <- c_value

  for (index in 2:(M - 1)) {
    f <- McCullagh_log_L(n, psi, delta, alpha, c)

    alpha[index] <- alpha[index] - delta1
    f0 <- McCullagh_log_L(n, psi, delta, alpha, c)

    alpha[index] <- alpha[index] + 2.0 * delta1
    f1 <- McCullagh_log_L(n, psi, delta, alpha, c)

    alpha[index] <- alpha[index] + delta1
    der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
    derivative <- McCullagh_derivative_log_l_wrt_alpha(n, index, psi, delta, alpha, c)
    if (derivative != 0.0) {
      expect_true(abs((der - derivative) / derivative) <= epsilon,
                  info=paste("logL alpha", index, der, derivative))
    } else {
        expect_true(abs(der - derivative) <= epsilon,
                    info=paste("logL alpha", index, der, derivative))
    }
  }
}
)


test_that("McCullagh_deriv_log_l_wrt_c", {
  c <- c_value

  f <- McCullagh_log_L(n, psi, delta, alpha, c)

  c <- c - delta1
  f0 <- McCullagh_log_L(n, psi, delta, alpha, c)

  c <- c + 2.0 * delta1
  f1 <- McCullagh_log_L(n, psi, delta, alpha, c)

  c <- c - delta1
  der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
  deriv <- McCullagh_derivative_log_l_wrt_c(n, psi, delta, alpha, c)
  if (der != 0.0) {
    expect_true(abs((der - deriv) / der) <= epsilon,
                info=paste("logL c", der, deriv))
  } else {
    expect_true(abs(der - deriv) <= epsilon, info=paste("der logL c", der, deriv))
  }
}
)


test_that("McCullagh_deriv_log_l_wrt_delta_vec", {
  M <- nrow(psi)
  c <- c_value

  f <- McCullagh_log_L(n, psi, delta_vec, alpha, c)
  for (k in 1:(M - 1)) {
    delta_vec[k] <- delta_vec[k] - delta1
    f0 <- McCullagh_log_L(n, psi, delta_vec, alpha, c)

    delta_vec[k] <- delta_vec[k] + 2.0 * delta1
    f1 <- McCullagh_log_L(n, psi, delta_vec, alpha, c)

    delta_vec[k] <- delta_vec[k] - delta1

    der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
    derivative <- McCullagh_derivative_log_l_wrt_delta_vec(n, k, psi, delta_vec, alpha, c)
    if (derivative != 0) {
      expect_true(abs((der - derivative) / derivative) <= epsilon,
                  info=paste("der logL wrt delta_vec", der, derivative))
    } else  {
      expect_true(abs(der - derivative) <= epsilon,
                  info=paste("der logL wrt delta_vec", der, derivative))
    }
  }
}
)


test_that("McCullagh_second_order_log_l_wrt_psi_2", {
  M = nrow(psi)
  c <- c_value

  for (i1 in 1:M) {
    for (j1 in i1:M) {
      for (i2 in 1:M) {
        for (j2 in i2:M) {
          f <- McCullagh_derivative_log_l_wrt_psi(n, i1, j1, psi, delta, alpha, c)

          psi[i2, j2] <- psi[i2, j2] - delta1
          psi[j2, i2] = psi[i2, j2]
          f0 <- McCullagh_derivative_log_l_wrt_psi(n, i1, j1, psi, delta, alpha, c)

          psi[i2, j2] <- psi[i2, j2] + 2.0 * delta1
          psi[j2, i2] <- psi[i2, j2]
          f1 <- McCullagh_derivative_log_l_wrt_psi(n, i1, j1, psi, delta, alpha, c)

          psi[i2, j2] <- psi[i2, j2] - delta1
          psi[j2, i2] <- psi[i2, j2]

          der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
          deriv <- McCullagh_second_order_log_l_wrt_psi_2(n, i1, j1, i2, j2, psi, delta, alpha, c)

          if (deriv == 0) {
            expect_true(abs(der - deriv) <= epsilon,
                        info=paste("d logL^2 d psi^2", i1, j1, i2, j2, der, deriv))
          } else {
            expect_true(abs((der - deriv) / deriv) <= epsilon,
                        info=paste("d^2 logL d psi^2", i1, j1, i2, j2, der, deriv))
          }
        }
      }
    }
  }
}
)


test_that("McCullagh_second_order_log_l_wrt_psi_delta", {
  M <- nrow(psi)
  c <- c_value
  N <- sum(n)

  for (i1 in 1:M) {
    for (j1 in i1: M) {
      f <- McCullagh_derivative_log_l_wrt_psi(n, i1, j1, psi, delta, alpha, c)

      delta <- delta - delta1
      f0 <- McCullagh_derivative_log_l_wrt_psi(n, i1, j1, psi, delta, alpha, c)

      delta <- delta + 2.0 * delta1
      f1 <- McCullagh_derivative_log_l_wrt_psi(n, i1, j1, psi, delta, alpha, c)

      delta <- delta - delta1

      der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
      deriv <- McCullagh_second_order_log_l_wrt_psi_delta(n, i1, j1, psi, delta, alpha, c)
      if (abs(deriv) > epsilon) {
        expect_true(abs((der - deriv) / deriv) <= epsilon,
                    info=paste("d^2 / d psi d delta-1", i1, j1, der, deriv))
      } else {
        expect_true(abs(der - deriv) <= epsilon,
                    info=paste("d^2 / d psi d delta-2", i1, j1, der, deriv))
      }
    }
  }
}
)


test_that("McCullagh_second_order_log_l_wrt_psi_alpha", {
  M <- nrow(psi)
  c <- c_value

  for (i1 in 1:M) {
    for (j1 in i1:M) {
      for (index in 2:(M - 1)) {
        f <- McCullagh_derivative_log_l_wrt_psi(n, i1, j1, psi, delta, alpha, c)

        alpha[index] <- alpha[index] - delta1
        f0 <- McCullagh_derivative_log_l_wrt_psi(n, i1, j1, psi, delta, alpha, c)

        alpha[index] <- alpha[index] + 2.0 * delta1
        f1 <- McCullagh_derivative_log_l_wrt_psi(n, i1, j1, psi, delta, alpha, c)

        alpha[index] <- alpha[index] - delta1

        der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
        deriv <- McCullagh_second_order_log_l_wrt_psi_alpha(n, i1, j1, index, psi, delta, alpha, c)
        if (deriv != 0.0) {
          expect_true(abs((der - deriv) / deriv) <= epsilon * 10.0,
                      info=paste("d^2 / d psi d alpha", i1, j1, index, der, deriv))
        } else {
          expect_true(abs(der - deriv) <= epsilon,
                      info=paste("d^2 / d psi d alpha", i1, j1, index, der, deriv))
        }
      }
    }
  }

  for (i1 in 1:M) {
    for (j1 in i1:M) {
      for (index in 2:(M - 1)) {
        f <- McCullagh_derivative_log_l_wrt_alpha(n, index, psi, delta, alpha, c)

        psi[i1, j1] <- psi[i1, j1] - delta1
        psi[j1, i1] <- psi[i1, j1]
        f0 <- McCullagh_derivative_log_l_wrt_alpha(n, index, psi, delta, alpha, c)

        psi[i1, j1] <- psi[i1, j1] + 2.0 * delta1
        psi[j1, i1] <- psi[i1, j1]
        f1 <- McCullagh_derivative_log_l_wrt_alpha(n, index, psi, delta, alpha, c)

        psi[i1, j1] <- psi[i1, j1] - delta1
        psi[j1, i1] <- psi[i1, j1]

        der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
        deriv <- McCullagh_second_order_log_l_wrt_psi_alpha(n, i1, j1, index, psi, delta, alpha, c)
        if (deriv != 0.0) {
          expect_true(abs((der - deriv) / deriv) <= epsilon * 10.0,
                      info=paste("d^2 / d psi d alpha", i1, j1, index, der, deriv))
        } else {
          expect_true(abs(der - deriv) <= epsilon,
                      info=paste("d^2 / d psi d alpha", i1, j1, index, der, deriv))
        }
      }
    }
  }
}
)


test_that("McCullagh_second_order_log_l_wrt_psi_c", {
  M <- nrow(psi)
  c <- c_value

  for (i1 in 1:M) {
    for (j1 in i1:M) {
      f <- McCullagh_derivative_log_l_wrt_c(n, psi, delta, alpha, c)

      psi[i1, j1] <- psi[i1, j1] - delta1
      psi[j1, i1] <- psi[i1, j1]
      f0 <- McCullagh_derivative_log_l_wrt_c(n, psi, delta, alpha, c)

      psi[i1, j1] <- psi[i1, j1] + 2.0 * delta1
      psi[j1, i1] <- psi[i1, j1]
      f1 <- McCullagh_derivative_log_l_wrt_c(n, psi, delta, alpha, c)

      psi[i1, j1] <- psi[i1, j1] - delta1
      psi[j1, i1] <- psi[i1, j1]

      der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
      deriv <- McCullagh_second_order_log_l_wrt_psi_c(n, i1, j1, psi, delta, alpha, c)
      if (abs(deriv) >= epsilon) {
        expect_true(abs((der - deriv) / deriv) <= epsilon,
                    info=paste("d^2 log_L d psi d c", i1, j1, der, deriv))
      } else {
        expect_true(abs(der - deriv) <= epsilon,
                    info=paste("d^2 log_L d psi d c", i1, j1, der, deriv))
      }
    }
  }
}
)


test_that("McCullagh_second_order_log_l_wrt_delta_2", {
  M <- nrow(psi)
  c <- c_value

  f <- McCullagh_derivative_log_l_wrt_delta(n, psi, delta, alpha, c)

  delta <- delta - delta1
  f0 <- McCullagh_derivative_log_l_wrt_delta(n, psi, delta, alpha, c)

  delta <- delta + 2.0 * delta1
  f1 <- McCullagh_derivative_log_l_wrt_delta(n, psi, delta, alpha, c)

  delta <- delta - delta1

  der = ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
  deriv = McCullagh_second_order_log_l_wrt_delta_2(n, psi, delta, alpha, c)
  expect_true(abs((der - deriv) / der) <= epsilon,
              info=paste("d^2 / d delta^2", der, deriv))
}
)


test_that("McCullagh_second_order_log_l_wrt_delta_alpha", {
  M <- nrow(psi)
  c <- c_value

  for (index in 2:(M - 1)) {
    f <- McCullagh_derivative_log_l_wrt_alpha(n, index, psi, delta, alpha, c)

    delta <- delta - delta1
    f0 <- McCullagh_derivative_log_l_wrt_alpha(n, index, psi, delta, alpha, c)

    delta <- delta + 2.0 * delta1
    f1 <- McCullagh_derivative_log_l_wrt_alpha(n, index, psi, delta, alpha, c)

    delta <- delta - delta1

    der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
    deriv <- McCullagh_second_order_log_l_wrt_delta_alpha(n, index, psi, delta, alpha, c)
    if (deriv != 0) {
      expect_true(abs((der - deriv) / deriv) <= epsilon,
                  info=paste("d^2 / d delta d alpha ", index, der, deriv))
    } else {
      expect_true(abs(der - deriv) <= epsilon,
                  info=paste("d^2 / d delta d alpha ", index, der, deriv))
    }
  }
}
)


test_that("McCullagh_second_order_log_l_wrt_delta_c", {
  M <- nrow(psi)
  c <- c_value

  f <- McCullagh_derivative_log_l_wrt_c(n, psi, delta, alpha, c)

  delta <- delta - delta1
  f0 <- McCullagh_derivative_log_l_wrt_c(n, psi, delta, alpha, c)

  delta <- delta + 2.0 * delta1
  f1 <- McCullagh_derivative_log_l_wrt_c(n, psi, delta, alpha, c)

  delta <- delta - delta1

  der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
  deriv <- McCullagh_second_order_log_l_wrt_delta_c(n, psi, delta, alpha, c)
  if (deriv != 0.0) {
    expect_true(abs((der - deriv) / deriv) <= epsilon,
                info=paste("d^2 log_L  d delta d c", der, deriv))
  } else {
    expect_true(abs(der - deriv) <= epsilon,
                info=paste("d^2 log_l d delta d c", der, deriv))
  }
}
)


test_that("McCullagh_second_order_log_l_wrt_alpha_2", {
  M <- nrow(psi)
  c <- c_value

  for (index1 in 2:(M - 1)) {
    for (index2 in 2:(M - 1)) {
      f <- McCullagh_derivative_log_l_wrt_alpha(n, index2, psi, delta, alpha, c)

      alpha[index1] <- alpha[index1] - delta1
      f0 <- McCullagh_derivative_log_l_wrt_alpha(n, index2, psi, delta, alpha, c)

      alpha[index1] <- alpha[index1] + 2.0 * delta1
      f1 <- McCullagh_derivative_log_l_wrt_alpha(n, index2, psi, delta, alpha, c)

      der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
      deriv <- McCullagh_second_order_log_l_wrt_alpha_2(n, index1, index2, psi, delta, alpha, c)
      if (deriv != 0.0) {
        expect_true(abs((der - deriv) / deriv) <= epsilon,
                    info=paste("d^logL / dalpha^2", index1, index2, der, deriv))
      } else {
        expect_true(abs(der - deriv) <= epsilon,
                    info=paste("d^logL / dalpha^2", index1, index2, der, deriv))
      }
    }
  }
}
)


test_that("McCullagh_second_order_log_l_wrt_alpha_c", {
  M <- nrow(psi)
  c <- c_value

  for (index in 2:(M - 1)) {
    f <- McCullagh_derivative_log_l_wrt_c(n, psi, delta, alpha, c)

    alpha[index] <- alpha[index] - delta1
    f0 <- McCullagh_derivative_log_l_wrt_c(n, psi, delta, alpha, c)

    alpha[index] <- alpha[index] + 2.0 * delta1
    f1 <- McCullagh_derivative_log_l_wrt_c(n, psi, delta, alpha, c)

    alpha[index] <- alpha[index] - delta1

    der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
    deriv <- McCullagh_second_order_log_l_wrt_alpha_c(n, index, psi, delta, alpha, c)
    if (deriv != 0.0) {
      expect_true(abs((der - deriv) / deriv) <= epsilon,
                  info=paste("d^2 log_L  d alpha d c", der, deriv))
    } else {
      expect_true(abs(der - deriv) <= epsilon,
                  info=paste("d^2 log_l d alpha d c", index, der, deriv))
    }
  }
}
)


test_that("McCullagh_second_order_log_l_wrt_c_2", {
  c <- c_value

  f <- McCullagh_derivative_log_l_wrt_c(n, psi, delta, alpha, c)

  c <- c - delta1
  f0 <- McCullagh_derivative_log_l_wrt_c(n, psi, delta, alpha, c)

  c <- c + 2.0 * delta1
  f1 <- McCullagh_derivative_log_l_wrt_c(n, psi, delta, alpha, c)

  c <- c - delta1
  der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
  deriv <- McCullagh_second_order_log_l_wrt_c_2(n, psi, delta, alpha, c)
  if (deriv != 0.0) {
    expect_true(abs((der - deriv) / deriv) <= epsilon,
                info=paste("d^2 log_L c^2", der, deriv))
  } else {
    expect_true(abs(der - deriv) <= epsilon,
                info=paste("d^2 log_l d c^2", der, deriv))
  }
}
)


test_that("McCullagh_second_order_log_l_wrt_psi_delta_vec", {
  M <- nrow(psi)
  c <- c_value
  N <- sum(n)

  for (k in 1:(M - 1)) {
    for (i1 in 1:M) {
      for (j1 in i1: M) {
        f <- McCullagh_derivative_log_l_wrt_delta_vec(n, k, psi, delta_vec, alpha, c)

        psi[i1, j1] <- psi[i1, j1] - delta1
        psi[j1, i1] <- psi[i1, j1]
        f0 <- McCullagh_derivative_log_l_wrt_delta_vec(n, k, psi, delta_vec, alpha, c)

        psi[i1, j1] <- psi[i1, j1] + 2.0 * delta1
        psi[j1, i1] <- psi[i1, j1]
        f1 <- McCullagh_derivative_log_l_wrt_delta_vec(n, k, psi, delta_vec, alpha, c)

        psi[i1, j1] <- psi[i1, j1] - delta1
        psi[j1, i1] <- psi[i1, j1]

        der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
        deriv <- McCullagh_second_order_log_l_wrt_psi_delta_vec(n, i1, j1, k, psi, delta_vec, alpha, c)
        if (abs(deriv) > epsilon) {
          expect_true(abs((der - deriv) / deriv) <= epsilon,
                      info=paste("d^2 / d psi d delta: 1", i1, j1, der, deriv))
        } else {
          expect_true(abs(der - deriv) <= epsilon,
                      info=paste("d^2 / d psi d delta: 2", i1, j1, der, deriv))
        }
      }
    }
  }
}
)


test_that("McCullagh_second_order_log_l_wrt_delta_vec_2", {
  M <- nrow(psi)
  c <- c_value

  for (k1 in 1:(M - 1)) {
    for (k2 in 1:(M - 1)) {
      f <- McCullagh_derivative_log_l_wrt_delta_vec(n, k2, psi, delta_vec, alpha, c)

      delta_vec[k1] <- delta_vec[k1] - delta1
      f0 <- McCullagh_derivative_log_l_wrt_delta_vec(n, k2, psi, delta_vec, alpha, c)

      delta_vec[k1] <- delta_vec[k1] + 2.0 * delta1
      f1 <- McCullagh_derivative_log_l_wrt_delta_vec(n, k2, psi, delta_vec, alpha, c)

      delta_vec[k1] <- delta_vec[k1] - delta1

      der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
      deriv <- McCullagh_second_order_log_l_wrt_delta_vec_2(n, k1, k2, psi, delta_vec, alpha, c)
      if (abs(deriv) > epsilon) {
        expect_true(abs((der - deriv) / deriv) <= epsilon,
                    info=paste("d^2 / d delta^2: 1", k1, k2, der, deriv))
      } else {
        expect_true(abs(der - deriv) <= epsilon,
                    info=paste("d^2 / d delta^2: 2", k1, k2, der, deriv))
      }
    }
  }
}
)


test_that("McCullagh_second_order_log_l_wrt_delta_vec_alpha", {
  M <- nrow(psi)
  c <- c_value
  N <- sum(n)

  for (k in 1:(M - 1)) {
    for (index in 2:(M - 1)) {
      f <- McCullagh_derivative_log_l_wrt_delta_vec(n, k, psi, delta_vec, alpha, c)

      alpha[index] <- alpha[index] - delta1
      f0 <- McCullagh_derivative_log_l_wrt_delta_vec(n, k, psi, delta_vec, alpha, c)

      alpha[index] <- alpha[index] + 2.0 * delta1
      f1 <- McCullagh_derivative_log_l_wrt_delta_vec(n, k, psi, delta_vec, alpha, c)

      alpha[index] <- alpha[index] - delta1

      der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
      deriv <- McCullagh_second_order_log_l_wrt_delta_vec_alpha(n, k, index, psi, delta_vec, alpha, c)
      if (abs(deriv) > epsilon) {
        expect_true(abs((der - deriv) / deriv) <= epsilon,
                    info=paste("d^2 / d delta_vec d alpha: 1", k, index, der, deriv))
      } else {
        expect_true(abs(der - deriv) <= epsilon,
                    info=paste("d^2 / d delta_vec d alpha: 2", k, index, der, deriv))
      }
    }
  }
}
)


test_that("McCullagh_second_order_log_l_wrt_delta_vec_c", {
  M <- nrow(psi)
  c <- c_value
  N <- sum(n)

  for (k in 1:(M - 1)) {
    f <- McCullagh_derivative_log_l_wrt_delta_vec(n, k, psi, delta_vec, alpha, c)

    c <- c - delta1
    f0 <- McCullagh_derivative_log_l_wrt_delta_vec(n, k, psi, delta_vec, alpha, c)

    c <- c + 2.0 * delta1
    f1 <- McCullagh_derivative_log_l_wrt_delta_vec(n, k, psi, delta_vec, alpha, c)

    c <- c - delta1

    der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
    deriv <- McCullagh_second_order_log_l_wrt_delta_vec_c(n, k, psi, delta_vec, alpha, c)
    if (abs(deriv) > epsilon) {
      expect_true(abs((der - deriv) / deriv) <= epsilon,
                  info=paste("d^2 / d delta d c: 1", k, der, deriv))
    } else {
      expect_true(abs(der - deriv) <= epsilon,
                  info=paste("d^2 / d delta d c: 2", k, der, deriv))
    }
  }
}
)


test_that("McCullagh_gradient_log_L", {
  M <- nrow(psi)
  c <- c_value

  gradient <- McCullagh_gradient_log_l(n, psi, delta, alpha, c)
  index <- 1
  for (i in 1:M) {
    for (j in i:M) {
      f <- McCullagh_log_L(n, psi, delta, alpha, c)

      psi[i, j] <- psi[i, j] - delta1
      psi[j, i] <- psi[i, j]
      f0 <- McCullagh_log_L(n, psi, delta, alpha, c)

      psi[i, j] <- psi[i, j] + 2.0 * delta1
      psi[j, i] <- psi[i, j]
      f1 <- McCullagh_log_L(n, psi, delta, alpha, c)

      psi[i, j] <- psi[i, j] - delta1
      psi[j, i] <- psi[i, j]
      der = ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0

      if (abs(der) <= 1.0e-4) {
        expect_true(abs(der - gradient[index]) <= epsilon,
                    info=paste("psi", index, der, gradient[index]))
      } else {
        expect_true(abs((der - gradient[index]) / gradient[index]) <= epsilon,
                    info=paste("psi", index, der, gradient[index]))
      }
      index <- index + 1
    }
  }

  # test delta
  for (k in 1:length(delta)) {
    f <- McCullagh_log_L(n, psi, delta, alpha, c)

    delta[k] <- delta - delta1
    f0 <- McCullagh_log_L(n, psi, delta, alpha, c)

    delta[k] <- delta + 2.0 * delta1
    f1 <- McCullagh_log_L(n, psi, delta, alpha, c)

    delta[k] <- delta - delta1
    der = ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
    if (gradient[index] != 0.0) {
      expect_true(abs((der - gradient[index]) / gradient[index]) <= epsilon,
                  info=paste("delta",der, gradient[index]))
    } else {
      expect_true(abs(der - gradient[index]) <= epsilon,
                  info=paste("delta",der, gradient[index]))
    }
    index <- index + 1
  }

  # test alpha
  for (i in 2:(M - 1)) {
    f <- McCullagh_log_L(n, psi, delta, alpha, c)

    alpha[i] <- alpha[i] - delta1
    f0 <- McCullagh_log_L(n, psi, delta, alpha, c)

    alpha[i] <- alpha[i] + 2.0 * delta1
    f1 <- McCullagh_log_L(n, psi, delta, alpha, c)

    alpha[i] <- alpha[i] - delta1
    der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
    if (der == 0.0) {
      expect_true(abs(der - gradient[index]) <= epsilon,
                  info=paste("alpha", index, i, der, gradient[index]))
    } else {
      expect_true(abs((der - gradient[index]) / gradient[index]) <= epsilon,
                  info=paste("alpha", index, i, der, index))
    }
    index <- index + 1
  }

  f <- McCullagh_log_L(n, psi, delta, alpha, c)

  c <- c - delta1
  f0 <- McCullagh_log_L(n, psi, delta, alpha, c)

  c <- c + 2.0 * delta1
  f1 <- McCullagh_log_L(n, psi, delta, alpha, c)

  c <- c - delta1
  der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
  if (der == 0.0) {
    expect_true(abs(der - gradient[index]) <= epsilon,
                info=paste("c", index, der, gradient[index]))
  } else {
    expect_true(abs((der - gradient[index]) / gradient[index]) <= epsilon,
                info=paste("c", index, der, gradient[index]))
  }
}
)


test_that("McCullagh_gradient_log_L_vec", {
  M <- nrow(psi)
  c <- c_value

  gradient <- McCullagh_gradient_log_l(n, psi, delta_vec, alpha, c)

  index <- 1
  for (i in 1:M) {
    for (j in i:M) {
      f <- McCullagh_log_L(n, psi, delta_vec, alpha, c)

      psi[i, j] <- psi[i, j] - delta1
      psi[j, i] <- psi[i, j]
      f0 <- McCullagh_log_L(n, psi, delta_vec, alpha, c)

      psi[i, j] <- psi[i, j] + 2.0 * delta1
      psi[j, i] <- psi[i, j]
      f1 <- McCullagh_log_L(n, psi, delta_vec, alpha, c)

      psi[i, j] <- psi[i, j] - delta1
      psi[j, i] <- psi[i, j]
      der = ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0

      if (abs(der) <= 1.0e-4) {
        expect_true(abs(der - gradient[index]) <= epsilon,
                    info=paste("psi", index, der, gradient[index]))
      } else {
        expect_true(abs((der - gradient[index]) / gradient[index]) <= epsilon,
                    info=paste("psi", index, der, gradient[index]))
      }
      index <- index + 1
    }
  }

  # test delta
  f <- McCullagh_log_L(n, psi, delta_vec, alpha, c)

  for (k in 1:(length(delta_vec))) {
    delta_vec[k] <- delta_vec[k] - delta1
    f0 <- McCullagh_log_L(n, psi, delta_vec, alpha, c)

    delta_vec[k] <- delta_vec[k] + 2.0 * delta1
    f1 <- McCullagh_log_L(n, psi, delta_vec, alpha, c)

    delta_vec[k] <- delta_vec[k] - delta1
    der = ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
    if (gradient[index] != 0.0) {
      expect_true(abs((der - gradient[index]) / gradient[index]) <= epsilon,
                  info=paste("delta",der, gradient[index]))
    } else {
      expect_true(abs(der - gradient[index]) <= epsilon,
                  info=paste("delta",der, gradient[index]))
    }
    index <- index + 1
  }

  # test alpha
  for (i in 2:(M - 1)) {
    f <- McCullagh_log_L(n, psi, delta_vec, alpha, c)

    alpha[i] <- alpha[i] - delta1
    f0 <- McCullagh_log_L(n, psi, delta_vec, alpha, c)

    alpha[i] <- alpha[i] + 2.0 * delta1
    f1 <- McCullagh_log_L(n, psi, delta_vec, alpha, c)

    alpha[i] <- alpha[i] - delta1
    der = ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
    if (der == 0.0) {
      expect_true(abs(der - gradient[index]) <= epsilon,
                  info=paste("alpha", index, i, der, gradient[index]))
    } else {
      expect_true(abs((der - gradient[index]) / gradient[index]) <= epsilon,
                  info=paste("alpha", index, i, der, index))
    }
    index <- index + 1
  }

  f <- McCullagh_log_L(n, psi, delta_vec, alpha, c)

  c <- c - delta1
  f0 <- McCullagh_log_L(n, psi, delta_vec, alpha, c)

  c <- c + 2.0 * delta1
  f1 <- McCullagh_log_L(n, psi, delta_vec, alpha, c)

  c <- c - delta1
  der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
  if (der == 0.0) {
    expect_true(abs(der - gradient[index]) <= epsilon,
                info=paste("c", index, der, gradient[index]))
  } else {
    expect_true(abs((der - gradient[index]) / gradient[index]) <= epsilon,
                info=paste("c", index, der, gradient[index]))
  }
}
)


test_that("McCullagh_hessian_log_l", {
  M <- nrow(psi)
  c <- c_value

  hessian <- McCullagh_hessian_log_l(n, psi, delta, alpha, c)
  M2 <- nrow(hessian)
  n_delta <- length(delta)

  # test that the hessian is symmetric
  for (i in 1:M2) {
    for (j in i:M2) {
      expect_true(abs(hessian[i, j] - hessian[j, i]) <= epsilon,
                  info=paste("Hessian is symmetric:", i, j, hessian[i, j], hessian[j, i]))
    }
  }

  # test the diagonal of the hessian for the section for psi.
  index <- 1
  for (i in 1:M) {
    for(j in i:M) {
      f <- McCullagh_derivative_log_l_wrt_psi(n, i, j, psi, delta, alpha, c)

      psi[i, j] <- psi[i, j] - delta1
      psi[j, i] <- psi[i, j]
      f0 <- McCullagh_derivative_log_l_wrt_psi(n, i, j, psi, delta, alpha, c)

      psi[i, j] <- psi[i, j] + 2.0 * delta1
      psi[j, i] <- psi[i, j]
      f1 <- McCullagh_derivative_log_l_wrt_psi(n, i, j, psi, delta, alpha, c)

      psi[i, j] <- psi[i, j] - delta1
      psi[j, i] <- psi[i, j]

      der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
      deriv <- hessian[index, index]
      if (deriv != 0.0) {
        expect_true(abs((der - deriv) / deriv) <= epsilon,
                    info=paste("psi", index, index, i, j, der, deriv))
      } else {
        expect_true(abs(der - deriv) <= epsilon,
                    info=paste(index, index, i, j, der, deriv))
      }
      index <- index + 1
    }
  }

  # test the off-diagonal of the hessian for psi
  index <- 1
  for (i1 in 1:M) {
    for (j1 in i1:M) {
      index2 <- 1
      for (i2 in 1:M) {
        for (j2 in i2: M) {
          f <- McCullagh_derivative_log_l_wrt_psi(n, i2, j2, psi, delta, alpha, c)

          psi[i1, j1] <- psi[i1, j1] - delta1
          psi[j1, i1] <- psi[i1, j1]
          f0 <- McCullagh_derivative_log_l_wrt_psi(n, i2, j2, psi, delta, alpha, c)

          psi[i1, j1] <- psi[i1, j1] + 2.0 * delta1
          psi[j1, i1] <- psi[i1, j1]
          f1 <- McCullagh_derivative_log_l_wrt_psi(n, i2, j2, psi, delta, alpha, c)

          psi[i1, j1] <- psi[i1, j1] - delta1
          psi[j1, i1] <- psi[i1, j1]

          der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
          deriv <- hessian[index, index2]

          if (deriv == 0.0) {
            expect_true(abs(der - deriv) <= epsilon,
                        info=paste("psi", i1, j1, i2, j2, index, index2, der, deriv))
          } else {
            expect_true(abs((der - deriv) / deriv) <= epsilon * 100.0,
                        info=paste("psi", i1, j1, i2, j2, index, index2, der, deriv))
          }
          index2 <- index2 + 1
        }
      }
      index <- index + 1
    }
  }

  # test diagonal of hessian for delta
  delta_position <- index
  for (k1 in 1:n_delta) {
    for (k2 in 1:n_delta) {
      f <- McCullagh_derivative_log_l_wrt_delta_vec(n, k2, psi, delta, alpha, c)

      delta[k1] <- delta[k1] - delta1
      f0 <- McCullagh_derivative_log_l_wrt_delta_vec(n, k2, psi, delta, alpha, c)

      delta[k1] <- delta[k1] + 2.0 * delta1
      f1 <- McCullagh_derivative_log_l_wrt_delta_vec(n, k2, psi, delta, alpha, c)

      delta <- delta - delta1

      der = ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
      deriv = hessian[delta_position, delta_position]

      expect_true(abs((der - deriv) / deriv) <= epsilon,
                  info=paste("delta", index, delta_position, der, deriv))
    }
  }
  index <- index + 1

  # test the off-diagonal elements for d psi d delta
  index = 1
  for (i in 1:M) {
    for (j in i:M) {
      f <- McCullagh_derivative_log_l_wrt_delta(n, psi, delta, alpha, c)

      psi[i, j] <- psi[i, j] - delta1
      psi[j, i] <- psi[i, j]
      f0 <- McCullagh_derivative_log_l_wrt_delta(n, psi, delta, alpha, c)

      psi[i, j] <- psi[i, j] + 2.0 * delta1
      psi[j, i] <- psi[i, j]
      f1 <- McCullagh_derivative_log_l_wrt_delta(n, psi, delta, alpha, c)

      psi[i, j] < psi[i, j] - delta1
      psi[j, i] = psi[i, j]

      der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
      deriv <- hessian[index, delta_position]
      if (abs(deriv) > epsilon) {
        expect_true(abs((der - deriv) / deriv) <= epsilon * 100.0,
                    info=paste("psi-delta", index, delta_position, der, deriv))

      } else {
        expect_true(abs(der - deriv) <= epsilon,
                    info=paste("psi-delta", index, delta_position, der, deriv))
      }

      index <- index + 1
    }
  }

  # test the off-diagonal elements for d psi d alpha
  index <- 1
  # i and j index psi
  for (i in 1:M) {
    for (j in i:M) {
      # alpha_index indexes alpha
      for (alpha_index in 2:(M - 1)) {
        f <- McCullagh_derivative_log_l_wrt_alpha(n, alpha_index, psi, delta, alpha, c)

        psi[i, j] <- psi[i, j] - delta1
        psi[j, i] <- psi[i, j]
        f0 <- McCullagh_derivative_log_l_wrt_alpha(n, alpha_index, psi, delta, alpha, c)

        psi[i, j] <- psi[i, j] + 2.0 * delta1
        psi[j, i] <- psi[i, j]
        f1 <- McCullagh_derivative_log_l_wrt_alpha(n, alpha_index, psi, delta, alpha, c)

        psi[i, j] <- psi[i, j] - delta1
        psi[j, i] <- psi[i, j]
        der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
        deriv <- hessian[index, delta_position + alpha_index - 1]

        if (deriv == 0.0) {
          expect_true(abs(der - deriv) <= epsilon,
                      info=paste("psi-alpha", index, delta_position + alpha_index - 1, der, deriv))
        } else {
          expect_true(abs((der - deriv) / deriv) <= epsilon * 100.0,
                      info=paste("psi-alpha", index, delta_position + alpha_index - 1, der, deriv))
        }
      }
      index <- index + 1
    }
  }

  # test off-diagonal elements for d psi d c
  index <- 1
  c_index <- alpha_index
  # i and j index psi
  for (i in 1:M) {
    for (j in i:M) {
      f <- McCullagh_derivative_log_l_wrt_c(n, psi, delta, alpha, c)

      psi[i, j] <- psi[i, j] - delta1
      psi[j, i] <- psi[i, j]
      f0 <- McCullagh_derivative_log_l_wrt_c(n, psi, delta, alpha, c)

      psi[i, j] <- psi[i, j] + 2.0 * delta1
      psi[j, i] <- psi[i, j]
      f1 <- McCullagh_derivative_log_l_wrt_c(n, psi, delta, alpha, c)

      psi[i, j] <- psi[i, j] - delta1
      psi[j, i] <- psi[i, j]

      der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
      deriv <- hessian[index, delta_position + c_index]
      if (abs(deriv) <= epsilon) {
        expect_true(abs(der - deriv) <= epsilon * 1000.0,
                    info=paste("psi-c", index, delta_position + c_index, der, deriv))
      } else {
        expect_true(abs((der - deriv) / deriv) <= epsilon * 10000.0,
                    info=paste("psi-c", index, delta_position + c_index, der, deriv))
      }
      index <- index + 1
    }
  }

  # test the off-diagonal elements for d delta d alpha
  for (alpha_index in 2:(M - 1)) {
    for (k in 1:n_delta) {
      f <- McCullagh_derivative_log_l_wrt_alpha(n, alpha_index, psi, delta, alpha, c)

      delta[k] <- delta[k] - delta1
      f0 <- McCullagh_derivative_log_l_wrt_alpha(n, alpha_index, psi, delta, alpha, c)

      delta[k] <- delta[k] + 2.0 * delta1
      f1 <- McCullagh_derivative_log_l_wrt_alpha(n, alpha_index, psi, delta, alpha, c)

      delta[k] <- delta[k] - delta1

      der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
      deriv <- hessian[delta_position + k - 1, delta_position + alpha_index - 1]
      expect_true(abs((der - deriv) / deriv) <= epsilon,
                  info=paste("delta-alpha", delta_position, delta_position + alpha_index - 1, der, deriv))
    }
  }

  # test off diagonal elements for d delta d c
  for (k in 1:n_delta) {
    f <- McCullagh_derivative_log_l_wrt_c(n, psi, delta, alpha, c)

    delta[k] <- delta[k] - delta1
    f0 <- McCullagh_derivative_log_l_wrt_c(n, psi, delta, alpha, c)

    delta[k] <- delta[k] + 2.0 * delta1
    f1 <- McCullagh_derivative_log_l_wrt_c(n, psi, delta, alpha, c)

    delta[k] <- delta[k] - delta1

    der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
    deriv <- hessian[delta_position + k - 1, delta_position + c_index]
    if (deriv == 0.0) {
      expect_true(abs(der - deriv) <= epsilon,
                  info=paste("delta-c", delta_position, delta_position + c_index, der, deriv))
    } else {
      expect_true(abs((der - deriv) / deriv) <= epsilon,
                  info=paste("delta-c", delta_position, delta_position + c_index, der, deriv))
    }
  }

  # test the mixed partial d alpha_index1 d alpha_index2
  for (alpha_index in 2:(M - 1)) {
    for (alpha_index2 in 2:(M - 1)) {
      f <- McCullagh_derivative_log_l_wrt_alpha(n, alpha_index2, psi, delta, alpha, c)

      alpha[alpha_index] <- alpha[alpha_index] - delta1
      f0 <- McCullagh_derivative_log_l_wrt_alpha(n, alpha_index2, psi, delta, alpha, c)

      alpha[alpha_index] <- alpha[alpha_index] + 2.0 * delta1
      f1 <- McCullagh_derivative_log_l_wrt_alpha(n, alpha_index2, psi, delta, alpha, c)

      alpha[alpha_index] <- alpha[alpha_index] - delta1

      der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
      deriv <- hessian[delta_position + alpha_index - 1, delta_position + alpha_index2 - 1]
      if (deriv != 0.0) {
        expect_true(abs((der - deriv) / deriv) <= epsilon * 10.0,
                    info=paste("alpha-alpha", delta_position + alpha_index - 1,
                              delta_position + alpha_index2 - 1, der, deriv))
      } else {
        expect_true(abs(der - deriv) <= epsilon,
                    info=paste("alpha-alpha", delta_position + alpha_index - 1,
                              delta_position + alpha_index2 - 1, der, deriv))
      }
    }
  }

  # text d^2 / d alpha d c
  for (alpha_index in 2:(M - 1)) {
    f <- McCullagh_derivative_log_l_wrt_alpha(n, alpha_index, psi, delta, alpha, c)

    c <- c - delta1
    f0 <- McCullagh_derivative_log_l_wrt_alpha(n, alpha_index, psi, delta, alpha, c)

    c <- c + 2.0 * delta1
    f1 <- McCullagh_derivative_log_l_wrt_alpha(n, alpha_index, psi, delta, alpha, c)

    c <- c - delta1

    der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
    deriv <- hessian[delta_position + alpha_index - 1, delta_position + c_index]
    expect_true(abs((der - deriv) / deriv) <= epsilon,
                info=paste("alpha-c", delta_position + alpha_index - 1, delta_position + c_index,
                          der, deriv))
  }

  # d^2 / d c^2
  f <- McCullagh_derivative_log_l_wrt_c(n, psi, delta, alpha, c)

  c <- c - delta1
  f0 <- McCullagh_derivative_log_l_wrt_c(n, psi, delta, alpha, c)

  c <- c + 2.0 * delta1
  f1 <- McCullagh_derivative_log_l_wrt_c(n, psi, delta, alpha, c)

  c <- c - delta1

  der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
  deriv <- hessian[delta_position + c_index, delta_position + c_index]
  if (deriv == 0) {
    expect_true(abs(der - deriv) < epsilon,
                info=paste("c^2", delta_position + c_index, delta_position + c_index, der, deriv))
  } else {
    expect_true(abs((der - deriv) / deriv) < epsilon,
                info=paste("c^2", delta_position + c_index, delta_position + c_index, der, deriv))
  }
}
)


test_that("McCullagh_hessian_log_l_vec", {
  M <- nrow(psi)
  c <- c_value

  hessian <- McCullagh_hessian_log_l(n, psi, delta_vec, alpha, c)
  M2 <- nrow(hessian)
  n_psi <- M * (M + 1) / 2
  n_delta <- length(delta_vec)
  n_alpha <- M - 2
  n_c <- 1
  delta_position <- n_psi + 1
  alpha_position <- delta_position + n_delta
  c_position <- alpha_position + n_alpha

  # test that the hessian is symmetric
  for (i in 1:M2) {
    for (j in i:M2) {
      expect_true(abs(hessian[i, j] - hessian[j, i]) <= epsilon,
                  info=paste("Hessian is symmetric:", i, j, hessian[i, j], hessian[j, i]))
    }
  }

  # test the diagonal of the hessian for the section for psi.
  index <- 1
  for (i in 1:M) {
    for(j in i:M) {
      f <- McCullagh_derivative_log_l_wrt_psi(n, i, j, psi, delta_vec, alpha, c)

      psi[i, j] <- psi[i, j] - delta1
      psi[j, i] <- psi[i, j]
      f0 <- McCullagh_derivative_log_l_wrt_psi(n, i, j, psi, delta_vec, alpha, c)

      psi[i, j] <- psi[i, j] + 2.0 * delta1
      psi[j, i] <- psi[i, j]
      f1 <- McCullagh_derivative_log_l_wrt_psi(n, i, j, psi, delta_vec, alpha, c)

      psi[i, j] <- psi[i, j] - delta1
      psi[j, i] <- psi[i, j]

      der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
      deriv <- hessian[index, index]
      if (deriv != 0.0) {
        expect_true(abs((der - deriv) / deriv) <= epsilon,
                    info=paste("psi", index, index, i, j, der, deriv))
      } else {
        expect_true(abs(der - deriv) <= epsilon,
                    info=paste(index, index, i, j, der, deriv))
      }
      index <- index + 1
    }
  }

  # test the off-diagonal of the hessian for psi
  index <- 1
  for (i1 in 1:M) {
    for (j1 in i1:M) {
      index2 <- 1
      for (i2 in 1:M) {
        for (j2 in i2: M) {
          f <- McCullagh_derivative_log_l_wrt_psi(n, i2, j2, psi, delta_vec, alpha, c)

          psi[i1, j1] <- psi[i1, j1] - delta1
          psi[j1, i1] <- psi[i1, j1]
          f0 <- McCullagh_derivative_log_l_wrt_psi(n, i2, j2, psi, delta_vec, alpha, c)

          psi[i1, j1] <- psi[i1, j1] + 2.0 * delta1
          psi[j1, i1] <- psi[i1, j1]
          f1 <- McCullagh_derivative_log_l_wrt_psi(n, i2, j2, psi, delta_vec, alpha, c)

          psi[i1, j1] <- psi[i1, j1] - delta1
          psi[j1, i1] <- psi[i1, j1]

          der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
          deriv <- hessian[index, index2]

          if (deriv == 0.0) {
            expect_true(abs(der - deriv) <= epsilon,
                        info=paste("psi", i1, j1, i2, j2, index, index2, der, deriv))
          } else {
            expect_true(abs((der - deriv) / deriv) <= epsilon * 100.0,
                        info=paste("psi", i1, j1, i2, j2, index, index2, der, deriv))
          }
          index2 <- index2 + 1
        }
      }
      index <- index + 1
    }
  }

  # test the off-diagonal elements for d psi d delta
  index <- 1
  for (i in 1:M) {
    for (j in i:M) {
      for (k in 1:n_delta) {
        f <- McCullagh_derivative_log_l_wrt_delta_vec(n, k, psi, delta_vec, alpha, c)

        psi[i, j] <- psi[i, j] - delta1
        psi[j, i] <- psi[i, j]
        f0 <- McCullagh_derivative_log_l_wrt_delta_vec(n, k, psi, delta_vec, alpha, c)

        psi[i, j] <- psi[i, j] + 2.0 * delta1
        psi[j, i] <- psi[i, j]
        f1 <- McCullagh_derivative_log_l_wrt_delta_vec(n, k, psi, delta_vec, alpha, c)

        psi[i, j] < psi[i, j] - delta1
        psi[j, i] = psi[i, j]

        der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
        deriv <- hessian[index, delta_position + k - 1]
        if (abs(deriv) > epsilon) {
          expect_true(abs((der - deriv) / deriv) <= epsilon * 100.0,
                      info=paste("psi-delta", index, delta_position + k - 1, der, deriv))
        } else {
          expect_true(abs(der - deriv) <= epsilon,
                      info=paste("psi-delta", index, delta_position + k - 1, der, deriv))
        }
      }
      index <- index + 1
    }
  }

  # test the off-diagonal elements for d psi d alpha
  index = 1
  # i and j index psi
  for (i in 1:M) {
    for (j in i:M) {
      # alpha_index indexes alpha
      for (alpha_index in 2:(M - 1)) {
        f <- McCullagh_derivative_log_l_wrt_alpha(n, alpha_index, psi, delta_vec, alpha, c)

        psi[i, j] <- psi[i, j] - delta1
        psi[j, i] <- psi[i, j]
        f0 <- McCullagh_derivative_log_l_wrt_alpha(n, alpha_index, psi, delta_vec, alpha, c)

        psi[i, j] <- psi[i, j] + 2.0 * delta1
        psi[j, i] <- psi[i, j]
        f1 <- McCullagh_derivative_log_l_wrt_alpha(n, alpha_index, psi, delta_vec, alpha, c)

        psi[i, j] <- psi[i, j] - delta1
        psi[j, i] <- psi[i, j]
        der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
        deriv <- hessian[index, delta_position + n_delta + alpha_index - 2]

        if (deriv == 0.0) {
          expect_true(abs(der - deriv) <= epsilon,
                      info=paste("psi-alpha", index, delta_position + n_delta + alpha_index - 2, der, deriv))
        } else {
          expect_true(abs((der - deriv) / deriv) <= epsilon * 100.0,
                      info=paste("psi-alpha", index, delta_position + n_delta + alpha_index - 2, der, deriv))
        }
      }
      index <- index + 1
    }
  }

  # test off-diagonal elements for d psi d c
  index <- 1
  # i and j index psi
  for (i in 1:M) {
    for (j in i:M) {
      f <- McCullagh_derivative_log_l_wrt_c(n, psi, delta_vec, alpha, c)

      psi[i, j] <- psi[i, j] - delta1
      psi[j, i] <- psi[i, j]
      f0 <- McCullagh_derivative_log_l_wrt_c(n, psi, delta_vec, alpha, c)

      psi[i, j] <- psi[i, j] + 2.0 * delta1
      psi[j, i] <- psi[i, j]
      f1 <- McCullagh_derivative_log_l_wrt_c(n, psi, delta_vec, alpha, c)

      psi[i, j] <- psi[i, j] - delta1
      psi[j, i] <- psi[i, j]

      der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
      deriv <- hessian[index, c_position]
      if (abs(deriv) <= epsilon) {
        expect_true(abs(der - deriv) <= epsilon * 1000.0,
                    info=paste("d^2 / d psi d c:", index, c_position, der, deriv))
      } else {
        expect_true(abs((der - deriv) / deriv) <= epsilon * 10000.0,
                    info=paste("d^2 / d psi d c:", index, c_position, der, deriv))
      }
      index <- index + 1
    }
  }

  # test the off-diagonal elements for d delta d alpha
  for (alpha_index in 2:(M - 1)) {
    for (k in 1:n_delta) {
      f <- McCullagh_derivative_log_l_wrt_alpha(n, alpha_index, psi, delta_vec, alpha, c)

      delta_vec[k] <- delta_vec[k] - delta1
      f0 <- McCullagh_derivative_log_l_wrt_alpha(n, alpha_index, psi, delta_vec, alpha, c)

      delta_vec[k] <- delta_vec[k] + 2.0 * delta1
      f1 <- McCullagh_derivative_log_l_wrt_alpha(n, alpha_index, psi, delta_vec, alpha, c)

      delta_vec[k] <- delta_vec[k] - delta1

      der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
      deriv <- hessian[delta_position + k - 1, delta_position + n_delta + alpha_index - 2]
      expect_true(abs((der - deriv) / deriv) <= epsilon,
                  info=paste("d^2 / d delta dalpha:", delta_position + k - 1,
                            delta_position + n_delta + alpha_index - 2, der, deriv))
    }
  }

  # test off-diagonal elements for d delta d c
  for (k in 1:n_delta) {
    f <- McCullagh_derivative_log_l_wrt_c(n, psi, delta_vec, alpha, c)

    delta_vec[k] <- delta_vec[k] - delta1
    f0 <- McCullagh_derivative_log_l_wrt_c(n, psi, delta_vec, alpha, c)

    delta_vec[k] <- delta_vec[k] + 2.0 * delta1
    f1 <- McCullagh_derivative_log_l_wrt_c(n, psi, delta_vec, alpha, c)

    delta_vec[k] <- delta_vec[k] - delta1

    der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
    deriv <- hessian[delta_position + k - 1, c_position]
    if (deriv == 0.0) {
      expect_true(abs(der - deriv) <= epsilon,
                  info=paste("delta-c", delta_position + k - 1, c_position, der, deriv))
    } else {
      expect_true(abs((der - deriv) / deriv) <= epsilon,
                  info=paste("d^2 / d delta d c:", delta_position + k - 1, c_index, der, deriv))
    }
  }

  # test the mixed partial d alpha_index1 d alpha_index2
  for (alpha_index in 2:(M - 1)) {
    for (alpha_index2 in 2:(M - 1)) {
      f <- McCullagh_derivative_log_l_wrt_alpha(n, alpha_index2, psi, delta_vec, alpha, c)

      alpha[alpha_index] <- alpha[alpha_index] - delta1
      f0 <- McCullagh_derivative_log_l_wrt_alpha(n, alpha_index2, psi, delta_vec, alpha, c)

      alpha[alpha_index] <- alpha[alpha_index] + 2.0 * delta1
      f1 <- McCullagh_derivative_log_l_wrt_alpha(n, alpha_index2, psi, delta_vec, alpha, c)

      alpha[alpha_index] <- alpha[alpha_index] - delta1

      der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
      deriv <- hessian[alpha_position + alpha_index - 2, alpha_position + alpha_index2 - 2]
      if (deriv != 0.0) {
        expect_true(abs((der - deriv) / deriv) <= epsilon * 10.0,
                    info=paste("alpha-alpha", alpha_position + alpha_index - 2,
                              alpha_position + alpha_index2 - 2, der, deriv))
      } else {
        expect_true(abs(der - deriv) <= epsilon,
                    info=paste("alpha-alpha", alpha_position + alpha_index - 2,
                              alpha_position + alpha_index2 - 2, der, deriv))
      }
    }
  }

  # text d^2 / d alpha d c
  for (alpha_index in 2:(M - 1)) {
    f <- McCullagh_derivative_log_l_wrt_alpha(n, alpha_index, psi, delta_vec, alpha, c)

    c <- c - delta1
    f0 <- McCullagh_derivative_log_l_wrt_alpha(n, alpha_index, psi, delta_vec, alpha, c)

    c <- c + 2.0 * delta1
    f1 <- McCullagh_derivative_log_l_wrt_alpha(n, alpha_index, psi, delta_vec, alpha, c)

    c <- c - delta1

    der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
    deriv <- hessian[alpha_position + alpha_index - 2, c_position]
    expect_true(abs((der - deriv) / deriv) <= epsilon,
                info=paste("alpha-c", alpha_position + alpha_index - 2, c_position, der, deriv))
  }

  # d^2 / d c^2
  f <- McCullagh_derivative_log_l_wrt_c(n, psi, delta_vec, alpha, c)

  c <- c - delta1
  f0 <- McCullagh_derivative_log_l_wrt_c(n, psi, delta_vec, alpha, c)

  c <- c + 2.0 * delta1
  f1 <- McCullagh_derivative_log_l_wrt_c(n, psi, delta_vec, alpha, c)

  c <- c - delta1

  der <- ((f - f0) / delta1 + (f1 - f) / delta1) / 2.0
  deriv <- hessian[c_position, c_position]
  if (deriv == 0) {
    expect_true(abs(der - deriv) < epsilon,
                info=paste("c^2", c_position, c_position, der, deriv))
  } else {
    expect_true(abs((der - deriv) / deriv) < epsilon,
                info=paste("c^2", delta_position + n_delta + c_index,
                          delta_position + n_delta + c_index, der, deriv))
  }
}
)


test_that("McCullagh_is_pi_invalid", {
    pi <- rep(1.0/14.0, 14)
    expect_true(!McCullagh_is_pi_invalid(pi), info="is_pi_invalid returns FALSE if all pi > 0.0")
    pi[5] <- 0.0
    expect_true(McCullagh_is_pi_invalid(pi), info="is_pi_invalid returns TRUE if one pi == 0.0")
    pi[5] <- -1.0 / 14.0
    expect_true(McCullagh_is_pi_invalid(pi), info="is_pi_invalid returns TRUE if one pi < 0.0")
  }
)
