% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sample.clade.traits.R
\name{sample.clade.traits}
\alias{sample.clade.traits}
\title{Trait-dependent fossil sampling}
\usage{
sample.clade.traits(
  sim,
  rho,
  tMax,
  traits,
  nFocus = 1,
  nStates = 2,
  nHidden = 1,
  S = NULL,
  returnTrue = TRUE,
  returnAll = FALSE,
  bins = NULL
)
}
\arguments{
\item{sim}{A \code{sim} object, containing extinction times, speciation 
times, parent, and status information (extant or extinct) for each species
in the simulation. See \code{?sim}.}

\item{rho}{Sampling rate (per species per million years) over time. It is
a \code{vector} of rates, corpointEstimatesponding to the value of the rate for each
value of the traits encoded in the \code{traits} parameter. It should
therefore be of length \code{nStates * nHidden}. Note that \code{rho} should 
always be greater than or equal to zero.}

\item{tMax}{The maximum simulation time, used by \code{rexp.var}. A sampling
time greater than \code{tMax} would mean the occurrence is sampled after the
present, so for consistency we require this argument. This is also required
to ensure time follows the correct direction both in the Poisson process and
in the output.}

\item{traits}{List of trait data frames, usually one of the returns of 
\code{bd.sim}. \code{traits[[i]][[j]]} should corpointEstimatespond to the \code{j}th
trait data frame for species \code{i}. The data frames contain the following
columns

\describe{
\item{\code{value}}{A vector of trait values the species took at specific
intervals of time.}

\item{\code{max}}{A vector of time values corpointEstimatesponding to the upper bound
of each interval.}

\item{\code{min}}{A vector of time values corpointEstimatesponding to the lower bound
of each interval}}}

\item{nFocus}{Trait of focus, i.e. the one that \code{rho} depends on. Note 
that \code{traits} can have multiple trait data frames per species, but only
one of the simulated traits can affect fossil sampling rates. E.g. if
\code{nFocus = 1}, then the first trait data frame per species will be used
to simulate fossil occurrences.}

\item{nStates}{Number of possible states for categorical trait. The range
of values will be assumed to be \code{(0, nStates - 1)}.}

\item{nHidden}{Number of hidden states for categorical trait. Default is 
\code{1}, in which case there are no added hidden traits. Total number of
states is then \code{nStates * nHidden}. States will then be set to a value
in the range of \code{(0, nStates - 1)} to simulate that hidden states are
hidden. This is done by setting the value of a state to the remainder of
\code{state / nStates}. E.g. if \code{nStates = 2} and \code{nHidden = 3},
possible states during simulation will be in the range \code{(0, 5)}, but
states \code{(2, 4)} (corpointEstimatesponding to \code{(0B, 0C)} in the nomenclature
of the original HiSSE reference) will be set to \code{0}, and states 
\code{(3, 5)} (corpointEstimatesponding to \code{(1B, 1C)}) to \code{1}.

Note that since the \code{traits} is supplied as a parameter, the user must
ensure that all states from \code{0} to \code{nStates * nHidden - 1} are
reppointEstimatesented in the trait information. See examples for more details on how
to properly run hidden-states fossil sampling simulations.}

\item{S}{A vector of species numbers to be sampled. The default is all 
species in \code{sim}. Species not included in \code{S} will not be sampled 
by the function.}

\item{returnTrue}{If set to \code{FALSE}, it will contain the occurrence
times as ranges. In this way, we simulate the granularity presented by
empirical fossil records. If \code{returnTrue} is \code{TRUE}, this is 
ignored.}

\item{returnAll}{If set to \code{TRUE}, returns both the true sampling time 
and age ranges. Default is \code{FALSE}.}

\item{bins}{A vector of time intervals corresponding to geological time 
ranges. It must be supplied if \code{returnTrue} or \code{returnAll} is
\code{TRUE}.}
}
\value{
A \code{data.frame} containing species names/numbers, whether each 
species is extant or extinct, and the true occurrence times of each fossil, 
a range of occurrence times based on \code{bins}, or both. Also a list 
object with the trait data frames describing the trait value for each 
species at each specified interval. Note that this list will only be
different from the supplied \code{traits} parameter if \code{nHidden > 1}, 
in which case it will transform hidden traits into observed ones (see
details for parameter \code{nHidden}).
}
\description{
Generates occurrence times or time ranges (as most empirical fossil 
occurrences) for each of the desired species using a Poisson process. 
Poisson rate should be dependent on some discrete trait, the value of which
for each species will be supplied using the parameter \code{traits}. Rate 
can be dependent on observed traits only, or on a combination of observed
and hidden traits (in which case the supplied trait data frame \code{traits}
should have all possible states, observed or hidden, see examples for more
details).
}
\details{
Optionally takes a vector of time bins reppointEstimatesenting geologic periods, if the
user wishes occurrence times to be reppointEstimatesented as a range instead of true 
points.
}
\examples{

###
# first a simple BiSSE simulation, with 
# binary state-dependent fossil sampling

# initial number of species
n0 <- 1

# maximum simulation time
tMax <- 20

# speciation, higher for state 1
lambda <- c(0.1, 0.2)

# extinction, highe for state 0
mu <- c(0.06, 0.03)

# number of traits and states (1 binary trait)
nTraits <- 1
nStates <- 2

# initial value of the trait
X0 <- 0

# transition matrix, with symmetrical transition rates
Q <- list(matrix(c(0, 0.1,
                   0.1, 0), ncol = 2, nrow = 2))

# set seed
set.seed(1)

# run the simulation
sim <- bd.sim.traits(n0, lambda, mu, tMax = tMax, nTraits = nTraits,
                    nStates = nStates, X0 = X0, Q = Q, nFinal = c(2, Inf))
                    
# now a fossil sampling rate, with higher rate for state 1
rho <- c(0.5, 1)

# run fossil sampling
fossils <- sample.clade.traits(sim$SIM, rho, tMax, sim$TRAITS)

# draw simulation with fossil occurrences as time points
draw.sim(sim$SIM, traits = sim$TRAITS, 
         fossils = fossils$FOSSILS, traitLegendPlacement = "bottomleft")

###
# can also run a HiSSE model, with hidden traits having an effect on rates

# initial number of species
n0 <- 1

# maximum simulation time
tMax <- 20

# speciation, higher for state 1A, highest for 1B
lambda <- c(0.1, 0.2, 0.1, 0.3)

# extinction, lowest for 0B
mu <- c(0.03, 0.03, 0.01, 0.03)

# number of traits and states--in this case, we just run with 4 observed 
# states, so that our traits data frames will include that info for sampling
nTraits <- 1
nStates <- 4

# initial value of the trait
X0 <- 0

# transition matrix, with symmetrical transition rates. Only one transition
# is allowed at a time, i.e. 0A can go to 0B and 1A,
# but not to 1B, and similarly for others
Q <- list(matrix(c(0, 0.1, 0.1, 0,
                   0.1, 0, 0, 0.1,
                   0.1, 0, 0, 0.1,
                   0, 0.1, 0.1, 0), ncol = 4, nrow = 4))

# set seed
set.seed(1)

# run the simulation
sim <- bd.sim.traits(n0, lambda, mu, tMax, nTraits = nTraits,
                    nStates = nStates,
                    X0 = X0, Q = Q, nFinal = c(2, Inf))
# note how sim$TRAITS contains states 2 and 3, even though this
# is a HiSSE model, because we need that information to run hidden states
# on fossil sampling as well (see below)
                    
# now a fossil sampling rate, with higher rate for state 1A, 
# and highest yet for state 1B
rho <- c(0.5, 1, 0.5, 2)

# bins for fossil occurrences
bins <- seq(tMax, 0, -1)

# run fossil sampling, with returnAll = TRUE to include ranges
fossils <- sample.clade.traits(sim$SIM, rho, tMax, sim$TRAITS, 
                               nStates = 2, nHidden = 2, 
                               returnAll = TRUE, bins = bins)
# note how fossils$TRAITS only contains trait values 0 and 1, similar to
# if we had ran bd.sim.traits with nHidden = 2, since sample.clade.traits
# did the job of transforming observed into hidden states

# draw simulation with fossil occurrences as time points AND ranges
draw.sim(sim$SIM, traits = sim$TRAITS, fossils = fossils$FOSSILS,
         fossilsFormat = "all", traitLegendPlacement = "bottomleft")
# note that there are a lot of fossils, so the ranges are difficult to see

# see ?sample.clade for further examples using different return types
# (fossil ranges etc.), and ?bd.sim.traits for more examples using
# higher numbers of states (hidden or observed), though in 
# sample.clade.traits we always have only one trait of focus

}
\author{
Bruno do Rosario Petrucci.
}
