#' Core Optimization Function
#'
#' @param choice Integer (1-6) selecting which built-in function to optimize
#' @param accuracy Desired precision
#' @param init_points Initial number of sampling points (minimum 2)
#' @param max_iter Maximum number of iterations
#' @param verbose Logical. Print progress.
#'
#' @return A list containing the optimization result
#' @export
optimize_main <- function(choice = 1, accuracy = 0.01, init_points = 50,
                          max_iter = 100, verbose = TRUE) {

  erf <- function(x) {
    2 * stats::pnorm(x * sqrt(2)) - 1
  }

  define_functions <- function() {
    list(
      function(x) 1 / (1 + exp(-x)),                                    # 1) Sigmoid
      function(x) tanh(x),                                              # 2) Tanh
      function(x) (1 / sqrt(2 * pi)) * exp(-x^2 / 2),                   # 3) PDF Normal
      function(x) if (x <= 0) Inf else (1 / (x * sqrt(2 * pi))) * exp(-log(x)^2 / 2),  # 4) PDF LogNormal
      function(x) 0.5 * (1 + erf(x / sqrt(2))),                         # 5) CDF Normal
      function(x) if (x <= 0) 0 else 0.5 * (1 + erf(log(x) / sqrt(2)))  # 6) CDF LogNormal
    )
  }

  domains <- list(
    c(-7, 7),        # 1) Sigmoid
    c(-4, 4),        # 2) Tanh
    c(-3.5, 3.5),    # 3) PDF Normal
    c(1e-6, 13.5),   # 4) PDF LogNormal
    c(-3.5, 3.5),    # 5) CDF Normal
    c(0, 22)         # 6) CDF LogNormal
  )

  # Validate choice
  if (!choice %in% 1:6) {
    stop("Choice must be an integer from 1 to 6.")
  }

  # Set function and domain based on choice
  funs <- define_functions()
  f <- funs[[choice]]
  domain <- domains[[choice]]

  if (length(domain) != 2 || domain[1] >= domain[2]) {
    stop("Invalid domain: must be a vector c(min, max) with min < max")
  }
  if (init_points < 2) {
    stop("init_points must be at least 2.")
  }

  lower <- domain[1]
  upper <- domain[2]

  # Initialize data
  data <- matrix(0, nrow = 10000, ncol = 2)
  for (i in 0:(init_points - 1)) {
    x <- lower + i * (upper - lower) / (init_points - 1)
    data[i + 1, ] <- c(x, f(x))
  }
  data_points <- init_points

  # Build initial PWL envelope
  envelope <- build_pwl_envelope(data[1:data_points, ], smallconst = accuracy)
  if (is.null(envelope) || is.null(envelope$PWL)) {
    stop("Failed to construct initial PWL envelope.")
  }

  pwl <- envelope$PWL
  breakpoints <- nrow(pwl)

  if (verbose) {
    cat("Initial Breakpoints =", breakpoints, "\n")
    cat("Initial Data Points =", data_points, "\n")
    cat("-------------------------------------------\n")
  }

  # Helper function: extract breakpoint coordinates from PWL segments
  extract_breakpoints <- function(pwl) {
    xs <- c(pwl[, 3], pwl[nrow(pwl), 4])  # segment start x's + last segment upper bound
    ys <- numeric(length(xs))
    for (i in seq_along(xs)) {
      x <- xs[i]
      seg_idx <- which(pwl[, 3] <= x & pwl[, 4] >= x)
      seg <- pwl[seg_idx[1], ]
      ys[i] <- seg[1] * x + seg[2]  # slope * x + intercept
    }
    coords <- cbind(x = xs, y = ys)
    return(coords)
  }

  start_time <- Sys.time()
  globaltime <- 0
  localtime <- 0

  for (k in 1:max_iter) {
    if (verbose) cat("\nIteration", k, "\n")

    start_local <- Sys.time()
    envelope <- build_pwl_envelope(data[1:data_points, ], smallconst = accuracy)
    end_local <- Sys.time()

    localtime <- localtime + as.numeric(difftime(end_local, start_local, units = "secs"))

    if (is.null(envelope) || is.null(envelope$PWL)) {
      stop("Envelope construction failed during iteration ", k)
    }

    pwl <- envelope$PWL
    breakpoints <- nrow(pwl)

    # Analyze global optima
    glob_minmax <- find_global_optima(pwl, f = f, tol = accuracy / 2)
    globby_min <- min(glob_minmax[, 2])
    globby_max <- max(glob_minmax[, 4])
    overall_max <- max(globby_max, -globby_min)

    if (verbose) cat("Overall Max =", overall_max, "\n")

    # Extract breakpoint coordinates
    bp_coords <- extract_breakpoints(pwl)

    # Check for convergence
    if (overall_max < accuracy + 1e-6) {
      duration <- difftime(Sys.time(), start_time, units = "secs")
      if (verbose) {
        cat("1: Converged after", k, "iterations.\n")
        cat("Final Overall Max =", overall_max, "\n")
        cat("Required Accuracy =", accuracy, "\n")
        cat("Number of breakpoints:", breakpoints, "\n")
        cat("Breakpoint coordinates (x, y):\n")
        print(bp_coords)
      }
      return(list(
        converged = TRUE,
        iterations = k,
        data = data[1:data_points, ],
        breakpoints = breakpoints,
        breakpoints_coords = bp_coords,
        duration = duration,
        local_time = localtime,
        global_time = globaltime,
        accuracy = overall_max,
        result = pwl
      ))
    }

    # Adaptive sampling
    threshold <- 4.0 / 6.0
    #threshold <- 5.0 / 6.0
    #threshold <- 1.0
    #tol <- 1e-4
    tol <- 1e-6

    data_new <- matrix(0, nrow = data_points * 3, ncol = 2)
    new_count <- 0

    for (i in 1:(data_points - 1)) {
      new_count <- new_count + 1
      data_new[new_count, ] <- data[i, ]

      for (b in 1:(breakpoints - 1)) {
        gmin_x <- glob_minmax[b, 1]; gmin_y <- glob_minmax[b, 2]
        gmax_x <- glob_minmax[b, 3]; gmax_y <- glob_minmax[b, 4]

        cond_min <- (-gmin_y > threshold * accuracy &&
                       gmin_x > data[i, 1] + tol &&
                       gmin_x > data_new[new_count, 1] + tol &&
                       gmin_x < data[i + 1, 1] - tol)

        cond_max <- (gmax_y > threshold * accuracy &&
                       gmax_x > data[i, 1] + tol &&
                       gmax_x > data_new[new_count, 1] + tol &&
                       gmax_x < data[i + 1, 1] - tol)

        if (cond_min) {
          new_count <- new_count + 1
          if(new_count > nrow(data_new)) {
            stop("Exceeded preallocated data_new rows during adaptive sampling (min).")
          }
          data_new[new_count, ] <- c(gmin_x, f(gmin_x))
        }
        if (cond_max) {
          new_count <- new_count + 1
          if(new_count > nrow(data_new)) {
            stop("Exceeded preallocated data_new rows during adaptive sampling (max).")
          }
          data_new[new_count, ] <- c(gmax_x, f(gmax_x))
        }
      }
    }

    new_count <- new_count + 1
    if(new_count > nrow(data_new)) {
      stop("Exceeded preallocated data_new rows when adding last point.")
    }
    data_new[new_count, ] <- data[data_points, ]

    if (new_count == data_points) {
      duration <- difftime(Sys.time(), start_time, units = "secs")
      if (verbose) {
        cat("2: Converged after", k, "iterations.\n")
        cat("Final Overall Max =", overall_max, "\n")
        cat("Required Accuracy =", accuracy, "\n")
        cat("Number of breakpoints:", breakpoints + 1, "\n")
        cat("Breakpoint coordinates (x, y):\n")
        print(bp_coords)
      }
      return(list(
        converged = TRUE,
        iterations = k,
        data = data[1:data_points, ],
        breakpoints = breakpoints,
        breakpoints_coords = bp_coords,
        duration = duration,
        local_time = localtime,
        global_time = globaltime,
        accuracy = overall_max,
        result = pwl
      ))
    }

    data_points <- new_count
    data <- data_new[1:data_points, , drop = FALSE]
  }

  # If not converged
  duration <- difftime(Sys.time(), start_time, units = "secs")
  if (verbose) {
    cat("Did not converge after max iterations.\n")
    cat("Number of breakpoints:", breakpoints + 1, "\n")
    cat("Breakpoint coordinates (x, y):\n")
    print(extract_breakpoints(pwl))
  }
  return(list(
    converged = FALSE,
    iterations = max_iter,
    data = data[1:data_points, ],
    breakpoints = breakpoints,
    breakpoints_coords = extract_breakpoints(pwl),
    duration = duration,
    local_time = localtime,
    global_time = globaltime,
    accuracy = overall_max,
    result = pwl
  ))
}
