% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ootb_sky_cie.R
\name{ootb_sky_cie}
\alias{ootb_sky_cie}
\title{Out-of-the-box CIE sky model and raster}
\usage{
ootb_sky_cie(
  r,
  z,
  a,
  m,
  bin,
  gs,
  min_spherical_dist = seq(0, 12, 3),
  method = c("Nelder-Mead", "BFGS", "CG", "SANN"),
  custom_sky_coef = NULL,
  parallel = TRUE
)
}
\arguments{
\item{r}{numeric \link[terra:SpatRaster-class]{terra::SpatRaster} of one layer. Typically, the blue
band of a a canopy photograph. Digital numbers should be linearly related
to radiance. See \code{\link[=read_caim_raw]{read_caim_raw()}} for details.}

\item{z}{\link[terra:SpatRaster-class]{terra::SpatRaster} generated with \code{\link[=zenith_image]{zenith_image()}}.}

\item{a}{\link[terra:SpatRaster-class]{terra::SpatRaster} generated with \code{\link[=azimuth_image]{azimuth_image()}}.}

\item{m}{logical \link[terra:SpatRaster-class]{terra::SpatRaster} with one layer. A binary mask with
\code{TRUE} for selected pixels.}

\item{bin}{logical \link[terra:SpatRaster-class]{terra::SpatRaster} with one layer. A binarized
hemispherical image. See \code{\link[=binarize_with_thr]{binarize_with_thr()}} for details.}

\item{gs}{\code{list} where each element is the output of
\code{\link[=sky_grid_segmentation]{sky_grid_segmentation()}}. See \emph{Examples} for guidance.}

\item{min_spherical_dist}{numeric vector. Values passed to
\code{\link[=rem_nearby_points]{rem_nearby_points()}}.}

\item{method}{character vector. Optimization methods for \code{\link[=fit_cie_model]{fit_cie_model()}}.}

\item{custom_sky_coef}{optional numeric vector of length five. If \code{NULL}
(default), the 15 standard CIE skies are tested as starting conditions. Use
this to avoid recomputing the initial step (depending only on \code{method})
when testing other inputs.}

\item{parallel}{logical vector of length one. If \code{TRUE}, operations are
executed in parallel.}
}
\value{
List with:
\describe{
\item{\code{rr_raster}}{numeric \link[terra:SpatRaster-class]{terra::SpatRaster}. Predicted relative radiance.}
\item{\code{model}}{list returned by \code{\link[=fit_cie_model]{fit_cie_model()}}. The optimal fit.}
\item{\code{model_validation}}{list returned by \code{\link[=validate_cie_model]{validate_cie_model()}}.}
\item{\code{dist_to_black}}{Value of \code{dist_to_black} used in \code{\link[=extract_sky_points]{extract_sky_points()}}
for the optimal fit.}
\item{\code{use_window}}{\code{logical}. Whether a window was used in \code{\link[=extract_rr]{extract_rr()}}
for the optimal fit.}
\item{\code{min_spherical_dist}}{Value of \code{min_dist} used in
\code{rem_nearby_points(space = "spherical")} for the optimal fit.}
\item{\code{sky_points}}{\code{data.frame} with columns \code{row} and \code{col}. Locations of
sky points.}
\item{\code{sun_row_col}}{\code{data.frame} with the estimated sun‑disk position in
image coordinates.}
\item{\code{g}}{Sky grid used for the optimal fit (as returned by \code{\link[=sky_grid_segmentation]{sky_grid_segmentation()}}).}
\item{\code{tested_grids}}{character vector describing the tested grid configurations.}
\item{\code{tested_distances}}{character vector of tested \code{min_dist} values in
\code{rem_nearby_points(space = "spherical")}.}
\item{\code{tested_methods}}{character vector of optimization methods tested in
\code{\link[=fit_cie_model]{fit_cie_model()}}.}
\item{\code{optimal_start}}{starting parameters selected after testing the 15 CIE skies.}
\item{\code{model_up}}{model fitted to relative radiance detected with the
two‑corner method, if that step succeeded; otherwise \code{NULL}.}
}
}
\description{
Fit and validate a CIE general sky model from a canopy photograph without
manual parameter tuning, and return the predicted raster.
}
\details{
Runs a full pipeline to fit a CIE sky model to radiance from a canopy
image:

\enumerate{
\item a preliminary estimate of sky digital numbers is attempted using the
two-corner method  aiming to start with a comprehensive sampling of the sky
vault (see \code{method = "detect_bg_dn"} of \code{\link[=apply_by_direction]{apply_by_direction()}}).
\item sky point extraction is performed with \code{\link[=extract_sky_points]{extract_sky_points()}}, using
information from a binary mask (\code{bin}) and post-filtering with
\code{\link[=rem_nearby_points]{rem_nearby_points()}} and \code{\link[=rem_outliers]{rem_outliers()}}.
\item relative radiance is computed with \code{\link[=extract_rr]{extract_rr()}} and fitted to CIE
sky models using \code{\link[=fit_cie_model]{fit_cie_model()}}, selecting the best among different
initial conditions and optimization methods.
\item model validation is performed via \code{\link[=validate_cie_model]{validate_cie_model()}}.
\item raster prediction with \code{\link[=cie_image]{cie_image()}}.
}
}
\note{
This function is part of a paper under preparation.
}
\examples{
\dontrun{
caim <- read_caim()
r <- caim$Blue
z <- zenith_image(ncol(caim), lens())
a <- azimuth_image(z)
m <- !is.na(z)

bin <- ootb_bin(caim, z, a, m, TRUE)

set.seed(7)
gs <- list(
  #high res
  sky_grid_segmentation(z, a, 2.25, first_ring_different = TRUE),
  sky_grid_segmentation(z, a, 2.8125, first_ring_different = TRUE),
  #medium res
  sky_grid_segmentation(z, a, 9, first_ring_different = TRUE),
  sky_grid_segmentation(z, a, 10, first_ring_different = TRUE),
  #low res
  sky_grid_segmentation(z, a, 15, first_ring_different = FALSE),
  sky_grid_segmentation(z, a, 18, first_ring_different = FALSE)
)

sky_cie <- ootb_sky_cie(r, z, a, m, bin, gs,
                        method = c("Nelder-Mead", "BFGS", "CG", "SANN"),
                        min_spherical_dist = seq(0, 12, 3),
                        parallel = TRUE)

sky_cie$rr_raster
plot(sky_cie$rr_raster)
sky_cie$model_validation$rmse
plot(sky_cie$model_validation$pred, sky_cie$model_validation$obs)
abline(0,1)

ratio <- r/sky_cie$rr_raster/sky_cie$model$rr$zenith_dn
plot(ratio)
plot(select_sky_region(ratio, 0.95, 1.05))
plot(select_sky_region(ratio, 1.15, max(ratio[], na.rm = TRUE)))

plot(bin)
points(sky_cie$sky_points$col,
       nrow(caim) - sky_cie$sky_points$row, col = 2, pch = 10)

}
}
