#' Interpolate in spherical space
#'
#' @description
#' Interpolate values from canopy photographs using inverse distance weighting
#' (IDW) with k-nearest neighbors, computing distances in spherical coordinates
#' that map the sky vault. Optionally blend with a model surface to fill voids.
#'
#' @details
#' Distances are great-circle distances on the sky vault. When `filling_source` is
#' provided, local IDW estimates are blended with that surface following Eq. 6
#' in \insertCite{Lang2010;textual}{rcaiman}. For efficiency, interpolation runs
#' on a temporary low-resolution grid of size `size`.
#'
#' @param sky_points `data.frame` returned by [extract_rr()]. If generated by
#'   other means, it must contain columns `row`, `col`, `z`, `a`, and `rr`,
#'   where the first four define geometry (degrees) and `rr` is the value to
#'   interpolate.
#' @param filling_source optional numeric [terra::SpatRaster-class] with one
#'   layer. Surface used to complement `rr` when neighbors are insufficient
#'   (e.g., output of [fit_cie_model()]). If `NULL` (default), no filling is
#'   applied.
#' @param w numeric vector of length one. Weight assigned to `filling_source`
#'   in the blend with local estimates (see Eq. 6 in
#'   \insertCite{Lang2010;textual}{rcaiman}).
#' @param k numeric vector of length one. Number of neighbors.
#' @param p numeric vector of length one. Inverse distance weighting exponent.
#' @param rule character vector of length one. Either `"any"` or `"all"`.
#'   With `"any"`, pixels within `angular_radius` of at least one sample are
#'   interpolated. With `"all"`, pixels are interpolated only if the `k`
#'   nearest neighbors lie within `angular_radius`. If `k = 1`, both are
#'   equivalent.
#' @param size numeric vector of length one. Number of rows and columns of the
#'   low-resolution grid used before resampling to full resolution.
#'
#' @inheritParams sky_grid_segmentation
#' @inheritParams interpolate_planar
#' @inheritParams rem_outliers
#'
#' @note
#' This function assumes that `sky_points` and the
#' [terra::SpatRaster-class] inputs are spatially aligned and share the same
#' geometry. No checks are performed to enforce this.
#'
#' @references \insertAllCited{}
#'
#' @return Numeric [terra::SpatRaster-class] with one layer of interpolated
#'   values and the geometry of `z`.
#'
#' @export
#'
#' @examples
#' \dontrun{
#' caim <- read_caim()
#' z <- zenith_image(ncol(caim), lens())
#' a <- azimuth_image(z)
#'
#' # Manual method following Lang et al. (2010)
#' path <- system.file("external/sky_points.csv",
#'                     package = "rcaiman")
#' sky_points <- read.csv(path)
#' sky_points <- sky_points[c("Y", "X")]
#' colnames(sky_points) <- c("row", "col")
#' head(sky_points)
#' plot(caim$Blue)
#' points(sky_points$col, nrow(caim) - sky_points$row, col = 2, pch = 10)
#'
#' # x11()
#' # plot(caim$Blue)
#' # sun_angles <- click(c(z, a), 1) %>% as.numeric()
#' sun_angles <- c(z = 49.5, a = 27.4) #taken with above lines then hardcoded
#'
#' sun_row_col <- row_col_from_zenith_azimuth(z, a,
#'                                            sun_angles["z"],
#'                                            sun_angles["a"])
#' points(sun_row_col[2], nrow(caim) - sun_row_col[1],
#'        col = "yellow", pch = 8, cex = 3)
#'
#' rr <- extract_rr(caim$Blue, z, a, sky_points)
#'
#' set.seed(7)
#' model <- fit_cie_model(rr, sun_angles,
#'                        general_sky_type = "Clear")
#'
#' sky_cie <- cie_image(z, a, model$sun_angles, model$coef)
#'
#' sky_rr <- interpolate_spherical(rr$sky_points, z, a,
#'                                 filling_source = sky_cie,
#'                                 w = 1,
#'                                 k = 10,
#'                                 p = 2,
#'                                 angular_radius = 20,
#'                                 rule = "any",
#'                                 size = 50)
#' plot(r/sky_rr/rr$zenith_dn)
#' }
interpolate_spherical <- function(sky_points, z, a,
                                  filling_source = NULL,
                                  w = 1,
                                  k = 3,
                                  p = 2,
                                  angular_radius = 20,
                                  rule = "any",
                                  size = 50
                              ) {

  # rr should be a list with a data frame component named `sky_points`
  if (!is.list(rr) || !("sky_points" %in% names(rr))) {
    stop("`rr` must be a list containing a component named `sky_points`.")
  }
  if (!is.data.frame(rr$sky_points)) {
    stop("`rr$sky_points` must be a data frame.")
  }
  # rr$sky_points should have the expected columns
  required_cols <- c("a", "z", "rr")
  if (!all(required_cols %in% names(rr$sky_points))) {
    stop(sprintf("`rr$sky_points` must contain columns %s.",
                 paste(sprintf('"%s"', required_cols), collapse = ", ")))
  }
  .check_r_z_a_m(NULL, z, a)
  if (!is.null(filling_source)) {
    .assert_single_layer(filling_source)
    .assert_same_geom(filling_source, z)
  } else {
    filling_source <- z
    filling_source[] <- NA
    w <- 0
  }
  .check_vector(w, "numeric", 1, sign = "nonnegative")
  .check_vector(k, "integerish", 1, sign = "positive")
  .check_vector(p, "numeric", 1, sign = "positive")
  .check_vector(angular_radius, "numeric", 1, sign = "positive")
  .assert_choice(rule, c("any", "all"))
  .check_vector(size, "integerish", 1, sign = "positive")


  sky_points2 <- expand.grid(row = seq(1, nrow(z), length.out = size) %>% round(),
                             col = seq(1, nrow(z), length.out = size) %>% round())
  sky_points2 <- extract_dn(z, sky_points2, use_window = FALSE)
  sky_points2 <- sky_points2[!is.na(sky_points2[,3]), c("row", "col")]

  # eq6 from Lang et al. 2010
  .eq6 <- function(w_k, b_k, W, M) {
    sum(c(sum(w_k * b_k), W * M), na.rm = TRUE) /
      (sum(w_k) + W)
  }

  sky_points2 <- extract_rr(z, z, a, sky_points2,
                                      no_of_points = NULL,
                                      use_window = FALSE)$sky_points
   sky_points[, c("z", "a")] <- .degree2radian(sky_points[, c("z", "a")])
  sky_points2[, c("z", "a")] <- .degree2radian(sky_points2[, c("z", "a")])
  angular_radius <- .degree2radian(angular_radius)
  .calculate_rr <- function(i) {
    spherical_distance <- calc_spherical_distance(sky_points$z,
                                                  sky_points$a,
                                                  sky_points2[i , "z"],
                                                  sky_points2[i , "a"])
    u <- terra::cellFromRowCol(z,
                               sky_points2[i , "row"],
                               sky_points2[i,  "col"])

    sky_points <- cbind(sky_points, chi = spherical_distance)
    sky_points <- sky_points[order(spherical_distance),]
    sky_points <- sky_points[1:k, ]
    m <- sky_points$chi <= angular_radius
    if (switch(rule, any = any(m), all = all(m))) {
      sky_points <- sky_points[m,]
      w_k <- 1 / sky_points$chi^p
      w_k <- w_k/max(w_k) #interpreting paragraph below eq6
      return(.eq6(w_k, sky_points[, "rr"], w, filling_source[u][,]))
    } else {
      return(filling_source[u][,])
    }
  }

  rr <- Map(.calculate_rr, seq_len(nrow(sky_points2))) %>% unlist()
  sky_points2$rr <- rr
  sky_points2 <- sky_points2[!is.na(rr),]
  sky_points2[, c("z", "a")] <- .radian2degree(sky_points2[, c("z", "a")])

  sky <- terra::rast(z)
  terra::ncol(sky) <- terra::nrow(sky) <- size
  xy <- terra::xyFromCell(z,
                          terra::cellFromRowCol(z, sky_points2$row,
                                                sky_points2$col))
  cell <- terra::cellFromXY(sky, xy)
  sky[cell] <- sky_points2$rr
  sky <- terra::focal(sky, 3, mean, na.policy = "only")
  sky <- terra::disagg(sky, terra::res(sky)[1], method = "bilinear")
  sky <- terra::resample(sky, z)
  sky[is.na(z)] <- NA

  names(sky) <- "Sky vault"
  sky
}
