#' Read UTF-8 File
#'
#' Reads a UTF-8 encoded file and returns its contents as a character vector.
#'
#' @param path Path to the file.
#' @return A character vector of lines.
#' @keywords internal
#' @noRd
#' @author
#' Yaoxiang Li
read_utf8 <- function(path) {
  con <- file(path, "r", encoding = "UTF-8")
  on.exit(close(con))
  readLines(con, warn = FALSE)
}

#' Split Lines
#'
#' Splits a string into individual lines.
#'
#' @param x A character vector.
#' @return A character vector of lines.
#' @keywords internal
#' @noRd
#' @author
#' Yaoxiang Li
split_lines <- function(x) {
  if (length(x) == 1) unlist(strsplit(x, "\n")) else x
}

#' Trim Whitespace
#'
#' Removes leading and trailing whitespace.
#'
#' @param x A character string.
#' @return The trimmed string.
#' @keywords internal
#' @noRd
#' @author
#' Yaoxiang Li
trim_space <- function(x) sub("^\\s+|\\s+$", "", x)

#' Merge Configurations
#'
#' Recursively merges two configuration lists, with values in the override
#' replacing those in the base configuration.
#'
#' @param base_cfg A list containing the base configuration.
#' @param override_cfg A list containing override values.
#' @importFrom utils modifyList
#' @return A merged configuration list.
#' @examples
#'
#' base_cfg <- get_config(system.file("extdata",
#' "config.yml",package = "rconf"), config_name = "default")
#' dev_cfg <- get_config(system.file("extdata",
#' "config.yml", package = "rconf"), config_name = "development")
#' combined_cfg <- merge_configs(base_cfg, dev_cfg)
#' print(combined_cfg)
#'
#' @export
merge_configs <- function(base_cfg, override_cfg) {
  modifyList(base_cfg, override_cfg)
}
