% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/compute_metrics.R
\name{compute_metrics}
\alias{compute_metrics}
\title{Compute spatially explicit watershed attributes for survey sites on streams}
\usage{
compute_metrics(
  metrics = c("lumped", "iFLO", "iFLS", "HAiFLO", "HAiFLS"),
  landuse,
  sites,
  out_fields,
  watersheds,
  flow_dir,
  flow_acc,
  streams,
  idwp = -1,
  percentage = TRUE,
  remove_streams = TRUE,
  max_memory = 300
)
}
\arguments{
\item{metrics}{A character vector. This vector specifies which metric(s) should be calculated. Your options are lumped, iFLO, iFLS, iEDO, iEDS, HAiFLO and HAiFLS. The default is to calculate the lumped, iFLO, iFLS, HAiFLO, and HAiFLS metrics.}

\item{landuse}{Names of landuse or landcover rasters in the current GRASS mapset. These can be continuous (e.g., percentage cover or NDVI) or binary, with a value of 1 for cells with a particular land use category and a value of 0 otherwise.}

\item{sites}{A set of survey sites in the current GRASS mapset.}

\item{out_fields}{A character vector of output field names to store the metrics. Note that \code{length(out_fields)} must be the same as \code{length(landuse) * length(metrics)}.}

\item{watersheds}{A vector of watershed raster names in the current GRASS mapset.}

\item{flow_dir}{Name of a flow direction raster produced by \code{derive_flow} in the current GRASS mapset.}

\item{flow_acc}{Name of a flow accumulation raster produced by \code{derive_flow} in the current GRASS mapset.}

\item{streams}{Name of a streams raster in the current GRASS mapset. The stream raster must have NoData values in cells that do not fall along the stream line. Optional if you are not asking for the iFLS, iEDS, and/or HAiFLS metrics.}

\item{idwp}{The inverse distance weighting parameter. Default is \code{-1}.}

\item{percentage}{A logical indicating whether the result should be expressed as a percentage. Defaults to \code{TRUE}. Set to \code{FALSE} if the landuse/landcover raster is continuous.}

\item{remove_streams}{A logical indicating whether cells falling on the stream line should be removed from iEDS, iFLS, and HAiFLS metrics. Defaults to \code{TRUE}, which is in line with the behaviour of IDWPLUS.}

\item{max_memory}{Max memory used in memory swap mode (MB). Defaults to \code{300}.}
}
\value{
A \code{sf} object of the snapped survey sites that also contains the computed landscape metrics.
}
\description{
Workhorse function for \code{rdwplus}. This function computes the spatially explicit landuse metrics in IDW-Plus (Peterson and Pearse, 2017).
}
\examples{
# Will only run if GRASS is running
# You should load rdwplus and initialise GRASS via the initGRASS function
if(check_running()){
# Retrieve paths to data sets
dem <- system.file("extdata", "dem.tif", package = "rdwplus")
lus <- system.file("extdata", "landuse.tif", package = "rdwplus")
sts <- system.file("extdata", "site.shp", package = "rdwplus")
stm <- system.file("extdata", "streams.shp", package = "rdwplus")

# Set environment 
set_envir(dem)

# Get other data sets (stream layer, sites, land use, etc.)
raster_to_mapset(lus)
vector_to_mapset(c(stm, sts))

# Reclassify streams
out_stream <- paste0(tempdir(), "/streams.tif")
rasterise_stream("streams", out_stream, TRUE)
reclassify_streams("streams.tif", "streams01.tif", overwrite = TRUE)

# Burn in the streams to the DEM
burn_in("dem.tif", "streams01.tif", "burndem.tif", overwrite = TRUE)

# Fill dem
fill_sinks("burndem.tif", "filldem.tif", "fd1.tif", "sinks.tif", overwrite = TRUE)

# Derive flow direction and accumulation grids
derive_flow("dem.tif", "fd.tif", "fa.tif", overwrite = T)

# Derive a new stream raster from the FA grid
derive_streams("dem.tif", "fa.tif", "new_stm.tif", "new_stm", min_acc = 200, overwrite = T)

# Snap sites to streams and flow accumulation
snap_sites("site", "new_stm.tif", "fa.tif", 2, "snapsite", T)

# Get watersheds
get_watersheds("snapsite", "fd.tif", "wshed.tif", T)

compute_metrics(
  metrics = c("lumped", "iFLO", "iEDO", "HAiFLO", "iFLS", "iEDS", "HAiFLS"),
  landuse = "landuse.tif",
  sites = "snapsite",
  out_fields = c("lumped", "iFLO", "iEDO", "HAiFLO", "iFLS", "iEDS", "HAiFLS"),
  watersheds = "wshed.tif",
  flow_dir = "fd.tif",
  flow_acc = "fa.tif",
  streams = "new_stm.tif",
  idwp = -1
)
}
}
\references{
Peterson, E.E. & Pearse, A.R. (2017). IDW-Plus: An ArcGIS toolset for calculating spatially explicit watershed attributes for survey sites. \emph{JAWRA}, \emph{53}(5), 1241-1249.
}
