% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/predict.R
\name{predict.sdmTMB}
\alias{predict.sdmTMB}
\title{Predict from an sdmTMB model}
\usage{
\method{predict}{sdmTMB}(
  object,
  newdata = NULL,
  type = c("link", "response"),
  se_fit = FALSE,
  re_form = NULL,
  re_form_iid = NULL,
  nsim = 0,
  sims_var = "est",
  model = c(NA, 1, 2),
  offset = NULL,
  mcmc_samples = NULL,
  return_tmb_object = FALSE,
  return_tmb_report = FALSE,
  return_tmb_data = FALSE,
  ...
)
}
\arguments{
\item{object}{A model fitted with \code{\link[=sdmTMB]{sdmTMB()}}.}

\item{newdata}{A data frame to make predictions on. This should be a data
frame with the same predictor columns as in the fitted data and a time
column (if this is a spatiotemporal model) with the same name as in the
fitted data.}

\item{type}{Should the \code{est} column be in link (default) or response space?}

\item{se_fit}{Should standard errors on predictions be calculated? Warning:
can be slow for large datasets or high-resolution projections when random
fields are included. For faster uncertainty estimation, either use
\code{re_form = NA} to exclude random fields or use the \code{nsim} argument to
simulate from the joint precision matrix.}

\item{re_form}{\code{NULL} to include all spatial/spatiotemporal random fields in
predictions. \code{~0} or \code{NA} for population-level predictions (predictions
from fixed effects only, marginalizing over random fields). Often used with
\code{se_fit = TRUE} to visualize marginal effects. Does not affect
\code{\link[=get_index]{get_index()}} calculations.}

\item{re_form_iid}{\code{NULL} to specify including all random intercepts in the
predictions. \code{~0} or \code{NA} for population-level predictions. No other
options (e.g., some but not all random intercepts) are implemented yet.
Only affects predictions with \code{newdata}. This \emph{does} affects \code{\link[=get_index]{get_index()}}.}

\item{nsim}{If \verb{> 0}, simulate from the joint precision matrix with \code{nsim}
draws. Returns a matrix of \code{nrow(newdata)} by \code{nsim} with each column
representing one draw of the linear predictor (in link space). Simulating
from the joint precision matrix accounts for uncertainty in both fixed and
random effects. Use this to derive uncertainty on predictions (e.g.,
\code{apply(x, 1, sd)}) or propagate uncertainty to derived quantities. This is
the fastest way to characterize spatial uncertainty with sdmTMB.}

\item{sims_var}{Experimental: Which TMB reported variable from the model
should be extracted from the joint precision matrix simulation draws?
Defaults to link-space predictions. Options include: \code{"omega_s"},
\code{"zeta_s"}, \code{"epsilon_st"}, and \code{"est_rf"} (as described below).
Other options will be passed verbatim.}

\item{model}{Which component to predict from delta/hurdle models when \code{nsim > 0} or \code{mcmc_samples} is supplied. \code{NA} (default) returns the combined
prediction from both components; \code{1} returns the binomial component only; \code{2}
returns the positive component only. Predictions are on the link or response
scale depending on \code{type}. For regular predictions (without simulation),
both components are returned. See the \href{https://sdmTMB.github.io/sdmTMB/articles/delta-models.html}{delta-model vignette}.}

\item{offset}{A numeric vector of optional offset values. If left at default
\code{NULL}, the offset is implicitly left at 0.}

\item{mcmc_samples}{See \code{extract_mcmc()} in the
\href{https://github.com/sdmTMB/sdmTMBextra}{sdmTMBextra} package for
more details and the
\href{https://sdmTMB.github.io/sdmTMB/articles/bayesian.html}{Bayesian vignette}.
If specified, the predict function will return a matrix of a similar form
as if \code{nsim > 0} but representing Bayesian posterior samples from the Stan
model.}

\item{return_tmb_object}{Logical. If \code{TRUE}, will include the TMB object in a
list format output. Necessary for the \code{\link[=get_index]{get_index()}} or \code{\link[=get_cog]{get_cog()}}
functions.}

\item{return_tmb_report}{Logical: return the output from the TMB
report? For regular prediction, this is all the reported variables
at the MLE parameter values. For \code{nsim > 0} or when \code{mcmc_samples}
is supplied, this is a list where each element is a sample and the
contents of each element is the output of the report for that sample.}

\item{return_tmb_data}{Logical: return formatted data for TMB? Used
internally.}

\item{...}{Not implemented.}
}
\value{
If \code{return_tmb_object = FALSE} (and \code{nsim = 0} and \code{mcmc_samples = NULL}):

A data frame:
\itemize{
\item \code{est}: Estimate in link space (everything included)
\item \code{est_non_rf}: Estimate from everything except random fields (fixed effects, random intercepts, time-varying effects, etc.)
\item \code{est_rf}: Estimate from all random fields combined
\item \code{omega_s}: Spatial random field (models consistent spatial patterns)
\item \code{zeta_s}: Spatially varying coefficient field (models how effects vary across space)
\item \code{epsilon_st}: Spatiotemporal random field (models spatial patterns that vary over time)
}

If \code{return_tmb_object = TRUE} (and \code{nsim = 0} and \code{mcmc_samples = NULL}):

A list:
\itemize{
\item \code{data}: The data frame described above
\item \code{report}: The TMB report on parameter values
\item \code{obj}: The TMB object returned from the prediction run
\item \code{fit_obj}: The original TMB model object
}

In this case, you likely only need the \code{data} element as an end user.
The other elements are included for other functions.

If \code{nsim > 0} or \code{mcmc_samples} is not \code{NULL}:

A matrix:
\itemize{
\item Columns represent samples
\item Rows represent predictions with one row per row of \code{newdata}
}
}
\description{
Make predictions from an \pkg{sdmTMB} model; can predict on the original or
new data.
}
\examples{
\dontshow{if (ggplot2_installed()) withAutoprint(\{ # examplesIf}

d <- pcod_2011
mesh <- make_mesh(d, c("X", "Y"), cutoff = 30) # a coarse mesh for example speed
m <- sdmTMB(
 data = d, formula = density ~ 0 + as.factor(year) + depth_scaled + depth_scaled2,
 time = "year", mesh = mesh, family = tweedie(link = "log")
)

# Predictions at original data locations -------------------------------

predictions <- predict(m)
head(predictions)

predictions$resids <- residuals(m) # randomized quantile residuals

library(ggplot2)
ggplot(predictions, aes(X, Y, col = resids)) + scale_colour_gradient2() +
  geom_point() + facet_wrap(~year)
hist(predictions$resids)
qqnorm(predictions$resids);abline(a = 0, b = 1)

# Predictions onto new data --------------------------------------------

qcs_grid_2011 <- replicate_df(qcs_grid, "year", unique(pcod_2011$year))
predictions <- predict(m, newdata = qcs_grid_2011)

\donttest{
# A short function for plotting our predictions:
plot_map <- function(dat, column = est) {
  ggplot(dat, aes(X, Y, fill = {{ column }})) +
    geom_raster() +
    facet_wrap(~year) +
    coord_fixed()
}

plot_map(predictions, exp(est)) +
  scale_fill_viridis_c(trans = "sqrt") +
  ggtitle("Prediction (fixed effects + all random effects)")

plot_map(predictions, exp(est_non_rf)) +
  ggtitle("Prediction (fixed effects and any time-varying effects)") +
  scale_fill_viridis_c(trans = "sqrt")

plot_map(predictions, est_rf) +
  ggtitle("All random field estimates") +
  scale_fill_gradient2()

plot_map(predictions, omega_s) +
  ggtitle("Spatial random effects only") +
  scale_fill_gradient2()

plot_map(predictions, epsilon_st) +
  ggtitle("Spatiotemporal random effects only") +
  scale_fill_gradient2()

# Visualizing a marginal effect ----------------------------------------

# See the visreg package or the ggeffects::ggeffect() or
# ggeffects::ggpredict() functions
# To do this manually:

nd <- data.frame(depth_scaled =
  seq(min(d$depth_scaled), max(d$depth_scaled), length.out = 100))
nd$depth_scaled2 <- nd$depth_scaled^2

# Because this is a spatiotemporal model, you'll need at least one time
# element. If time isn't also a fixed effect then it doesn't matter what you pick:
nd$year <- 2011L # L: integer to match original data
p <- predict(m, newdata = nd, se_fit = TRUE, re_form = NA)
ggplot(p, aes(depth_scaled, exp(est),
  ymin = exp(est - 1.96 * est_se), ymax = exp(est + 1.96 * est_se))) +
  geom_line() + geom_ribbon(alpha = 0.4)

# Plotting marginal effect of a spline ---------------------------------

m_gam <- sdmTMB(
 data = d, formula = density ~ 0 + as.factor(year) + s(depth_scaled, k = 5),
 time = "year", mesh = mesh, family = tweedie(link = "log")
)
if (require("visreg", quietly = TRUE)) {
  visreg::visreg(m_gam, "depth_scaled")
}

# or manually:
nd <- data.frame(depth_scaled =
  seq(min(d$depth_scaled), max(d$depth_scaled), length.out = 100))
nd$year <- 2011L
p <- predict(m_gam, newdata = nd, se_fit = TRUE, re_form = NA)
ggplot(p, aes(depth_scaled, exp(est),
  ymin = exp(est - 1.96 * est_se), ymax = exp(est + 1.96 * est_se))) +
  geom_line() + geom_ribbon(alpha = 0.4)

# Forecasting ----------------------------------------------------------
mesh <- make_mesh(d, c("X", "Y"), cutoff = 15)

unique(d$year)
m <- sdmTMB(
  data = d, formula = density ~ 1,
  spatiotemporal = "AR1", # using an AR1 to have something to forecast with
  extra_time = 2019L, # `L` for integer to match our data
  spatial = "off",
  time = "year", mesh = mesh, family = tweedie(link = "log")
)

# Add a year to our grid:
grid2019 <- qcs_grid_2011[qcs_grid_2011$year == max(qcs_grid_2011$year), ]
grid2019$year <- 2019L # `L` because `year` is an integer in the data
qcsgrid_forecast <- rbind(qcs_grid_2011, grid2019)

predictions <- predict(m, newdata = qcsgrid_forecast)
plot_map(predictions, exp(est)) +
  scale_fill_viridis_c(trans = "log10")
plot_map(predictions, epsilon_st) +
  scale_fill_gradient2()

# Estimating local trends ----------------------------------------------

d <- pcod
d$year_scaled <- as.numeric(scale(d$year))
mesh <- make_mesh(pcod, c("X", "Y"), cutoff = 25)
m <- sdmTMB(data = d, formula = density ~ depth_scaled + depth_scaled2,
  mesh = mesh, family = tweedie(link = "log"),
  spatial_varying = ~ 0 + year_scaled, time = "year", spatiotemporal = "off")
nd <- replicate_df(qcs_grid, "year", unique(pcod$year))
nd$year_scaled <- (nd$year - mean(d$year)) / sd(d$year)
p <- predict(m, newdata = nd)

plot_map(subset(p, year == 2003), zeta_s_year_scaled) + # pick any year
  ggtitle("Spatial slopes") +
  scale_fill_gradient2()

plot_map(p, est_rf) +
  ggtitle("Random field estimates") +
  scale_fill_gradient2()

plot_map(p, exp(est_non_rf)) +
  ggtitle("Prediction (fixed effects only)") +
  scale_fill_viridis_c(trans = "sqrt")

plot_map(p, exp(est)) +
  ggtitle("Prediction (fixed effects + all random effects)") +
  scale_fill_viridis_c(trans = "sqrt")
}
\dontshow{\}) # examplesIf}
}
