#' Search Reproducibility Manager
#'
#' @description
#' A comprehensive system for managing and validating the reproducibility
#' of systematic review search strategies and analyses.
#'
#' @details
#' The ReproducibilityManager class provides tools for:
#' \itemize{
#'   \item Creating reproducible search packages
#'   \item Validating reproducibility of existing packages
#'   \item Generating audit trails
#'   \item Ensuring transparency and reproducibility in evidence synthesis
#' }
#'
#' @section Methods:
#' \describe{
#'   \item{\code{new()}}{Initialize a new ReproducibilityManager instance}
#'   \item{\code{create_repro_package(search_strategy, results, analysis_config)}}{Create reproducible search package}
#'   \item{\code{validate_repro(package_path)}}{Validate reproducibility of existing package}
#'   \item{\code{gen_audit_trail(search_analysis)}}{Generate audit trail}
#' }
#'
#' @examples
#' # Create reproducibility manager
#' manager <- ReproducibilityManager$new()
#'
#' # Create sample search strategy
#' search_strategy <- list(
#'   terms = c("systematic review", "meta-analysis"),
#'   databases = c("PubMed", "Embase"),
#'   timestamp = Sys.time(),
#'   date_range = as.Date(c("2020-01-01", "2023-12-31"))
#' )
#'
#' # Create sample search results
#' search_results <- data.frame(
#'   id = paste0("article_", 1:20),
#'   title = paste("Research Study", 1:20),
#'   abstract = paste("Abstract for study", 1:20),
#'   source = "Journal of Research",
#'   date = Sys.Date() - sample(1:365, 20, replace = TRUE),
#'   stringsAsFactors = FALSE
#' )
#'
#' # Create sample analysis configuration
#' analysis_config <- list(
#'   gold_standard = paste0("article_", sample(1:20, 5)),
#'   method = "precision_recall",
#'   parameters = list(threshold = 0.8)
#' )
#'
#' # Create reproducible package (writes to tempdir())
#' package_path <- manager$create_repro_package(
#'   search_strategy = search_strategy,
#'   results = search_results,
#'   analysis_config = analysis_config
#' )
#'
#' print(paste("Package created at:", package_path))
#'
#' # Generate audit trail (create mock analyzer object for demonstration)
#' mock_analysis <- list(
#'   search_results = search_results,
#'   metadata = list(timestamp = Sys.time())
#' )
#' class(mock_analysis) <- "mock_analyzer"
#'
#' audit_trail <- manager$gen_audit_trail(mock_analysis)
#' print("Audit trail generated successfully")
#'
#' @export
ReproducibilityManager <- R6::R6Class(
  "ReproducibilityManager",
  public = list(
    #' @description
    #' Creates a new ReproducibilityManager instance for managing search reproducibility.
    #' Sets up necessary configurations and validates system requirements.
    #' @return No return value, called for side effects (initialization)
    initialize = function() {
      # Initialize configurations
      private$supported_formats <- c("rds", "csv", "json")
      private$required_packages <- c("digest", "glue", "rmarkdown")

      # Check system requirements
      private$check_system_requirements()

      invisible(self)
    },

    #' Create reproducible search package
    #' @param search_strategy Search strategy object
    #' @param results Search results
    #' @param analysis_config Analysis configuration
    #' @return Path to reproducibility package
    create_repro_package = function(search_strategy, results, analysis_config) {
      package_dir <- file.path(tempdir(), "repro_package")

      # Create directory structure
      dir.create(file.path(package_dir, "data"), recursive = TRUE)
      dir.create(file.path(package_dir, "code"), recursive = TRUE)
      dir.create(file.path(package_dir, "reports"), recursive = TRUE)

      # Save search strategy
      saveRDS(search_strategy, file.path(package_dir, "data", "search_strategy.rds"))

      # Save results
      write.csv(results, file.path(package_dir, "data", "search_results.csv"), row.names = FALSE)

      # Save analysis configuration
      saveRDS(analysis_config, file.path(package_dir, "data", "analysis_config.rds"))

      # Generate reproduction script
      private$generate_reproduction_script(package_dir, search_strategy, analysis_config)

      # Create manifest
      private$create_manifest(package_dir)

      # Create README
      private$create_readme(package_dir, search_strategy)

      return(package_dir)
    },

    #' Validate reproducibility of existing package
    #' @param package_path Path to reproducibility package
    #' @return Validation results
    validate_repro = function(package_path) {
      # Load original data
      original_strategy <- readRDS(file.path(package_path, "data", "search_strategy.rds"))
      original_results <- read.csv(file.path(package_path, "data", "search_results.csv"), stringsAsFactors = FALSE)

      # Re-execute search
      reproduced_results <- private$re_execute_search(original_strategy)

      # Compare results
      comparison <- private$compare_results(original_results, reproduced_results)

      # Generate validation report
      validation_report <- list(
        timestamp = Sys.time(),
        original_date = original_strategy$timestamp,
        comparison = comparison,
        reproducible = comparison$match_rate > 0.95
      )

      class(validation_report) <- "reproducibility_validation"
      return(validation_report)
    },

    #' Generate audit trail
    #' @param search_analysis SearchAnalyzer object
    #' @return Audit trail object
    gen_audit_trail = function(search_analysis) {
      audit_trail <- list(
        session_info = sessionInfo(),
        package_versions = private$get_pkg_versions(),
        system_info = Sys.info(),
        execution_time = search_analysis$metadata$timestamp,
        git_commit = private$get_git_commit(),
        checksum = private$calculate_data_checksum(search_analysis$search_results)
      )

      class(audit_trail) <- "audit_trail"
      return(audit_trail)
    }
  ),

  private = list(
    supported_formats = NULL,
    required_packages = NULL,

    check_system_requirements = function() {
      # Check if required packages are available using requireNamespace
      missing_packages <- c()
      for (pkg in private$required_packages) {
        if (!requireNamespace(pkg, quietly = TRUE)) {
          missing_packages <- c(missing_packages, pkg)
        }
      }

      if (length(missing_packages) > 0) {
        warning("Missing required packages for reproducibility: ",
                paste(missing_packages, collapse = ", "),
                "\nTo install: install.packages(c(",
                paste0('"', missing_packages, '"', collapse = ", "), "))")
      }

      invisible(TRUE)
    },

    generate_reproduction_script = function(package_dir, strategy, config) {
      # Using base R paste instead of glue::glue
      script_content <- paste0(
        "# Reproduction Script for Search Strategy Analysis\n",
        "# Generated: ", Sys.time(), "\n\n",
        "library(searchAnalyzeR)\n",
        "# Note: Using base R functions instead of tidyverse for better compatibility\n\n",
        "# Load original data\n",
        "search_strategy <- readRDS(\"data/search_strategy.rds\")\n",
        "search_results <- read.csv(\"data/search_results.csv\", stringsAsFactors = FALSE)\n",
        "analysis_config <- readRDS(\"data/analysis_config.rds\")\n\n",
        "# Initialize analyzer\n",
        "analyzer <- SearchAnalyzer$new(\n",
        "  search_results = search_results,\n",
        "  gold_standard = analysis_config$gold_standard,\n",
        "  search_strategy = search_strategy\n",
        ")\n\n",
        "# Calculate metrics\n",
        "metrics <- analyzer$calculate_metrics()\n\n",
        "# Generate visualizations\n",
        "plots <- list(\n",
        "  overview = analyzer$visualize_performance(\"overview\"),\n",
        "  precision_recall = analyzer$visualize_performance(\"precision_recall\")\n",
        ")\n\n",
        "# Generate report\n",
        "reporter <- PRISMAReporter$new()\n",
        "report_path <- reporter$generate_report(analyzer, output_format = \"html\")\n\n",
        "cat(\"Analysis completed successfully!\\n\")\n",
        "cat(\"Report saved to:\", report_path, \"\\n\")\n"
      )

      writeLines(script_content, file.path(package_dir, "code", "reproduce_analysis.R"))
    },

    create_manifest = function(package_dir) {
      files <- list.files(package_dir, recursive = TRUE, full.names = TRUE)

      # Using base R instead of lapply + do.call(rbind, ...)
      manifest_list <- list()
      for (i in seq_along(files)) {
        file <- files[i]
        manifest_list[[i]] <- data.frame(
          file = file,
          size = file.info(file)$size,
          modified = file.info(file)$mtime,
          checksum = digest::digest(file, file = TRUE),
          stringsAsFactors = FALSE
        )
      }

      manifest <- do.call(rbind, manifest_list)
      write.csv(manifest, file.path(package_dir, "MANIFEST.csv"), row.names = FALSE)
    },

    create_readme = function(package_dir, strategy) {
      # Using base R paste instead of glue::glue
      readme_content <- paste0(
        "# Reproducible Search Strategy Package\n\n",
        "Generated: ", Sys.time(), "\n",
        "Strategy: ", paste(strategy$terms, collapse = ", "), "\n\n",
        "## Contents\n\n",
        "- `data/`: Original data files\n",
        "- `code/`: Reproduction scripts\n",
        "- `reports/`: Generated reports\n\n",
        "## Usage\n\n",
        "Run the reproduction script:\n",
        "```r\n",
        "source(\"code/reproduce_analysis.R\")\n",
        "```\n"
      )

      writeLines(readme_content, file.path(package_dir, "README.md"))
    },

    re_execute_search = function(strategy) {
      # Placeholder for re-executing search
      # In practice, this would re-run the actual search
      warning("Search re-execution not implemented - returning placeholder results")
      return(data.frame(id = character(0), stringsAsFactors = FALSE))
    },

    compare_results = function(original, reproduced) {
      # Compare original and reproduced results
      list(
        original_count = nrow(original),
        reproduced_count = nrow(reproduced),
        match_rate = length(intersect(original$id, reproduced$id)) / max(nrow(original), 1)
      )
    },

    get_pkg_versions = function() {
      relevant_packages <- c("searchAnalyzeR", "dplyr", "ggplot2")

      # Using requireNamespace instead of installed.packages() for better performance
      versions <- sapply(relevant_packages, function(pkg) {
        if (requireNamespace(pkg, quietly = TRUE)) {
          tryCatch({
            as.character(utils::packageVersion(pkg))
          }, error = function(e) {
            "Version unavailable"
          })
        } else {
          "Not installed"
        }
      })

      return(versions)
    },

    get_git_commit = function() {
      # Try to get git commit hash
      tryCatch({
        system("git rev-parse HEAD", intern = TRUE)
      }, error = function(e) {
        "Git not available or not a git repository"
      })
    },

    calculate_data_checksum = function(data) {
      # Calculate checksum of the data
      digest::digest(data, algo = "md5")
    }
  )
)
