#' @title Deseasonalize a Time Series Given Seasonal Indices
#' @param data   A numeric vector or `ts` object of original data.
#' @param indices A numeric vector (or data.frame column) of seasonal indices of length `sl`.
#' @param sl     A positive integer giving the seasonal period.
#' @param season_labels Optional character vector of length `sl` giving labels for each season.
#' @return A data.frame with columns: Observation (index), Season (label), Original (rounded original data), Deseasonalized (rounded).
#' @export
#' @examples
#' data <- as.ts(rnorm(48, mean = 100, sd = 48), frequency = 12)
#' sl <- 12
#' si <- SI(data, sl)
#' deseasonalized <- deseasonalize(data, si$Seasonal_Index, sl)
deseasonalize <- function(data, indices, sl, season_labels = NULL) {
  n <- length(data)
  season_index <- rep(seq_len(sl), length.out = n)
  s_index <- as.numeric(indices)
  deseasonalized <- data / s_index[season_index]
  if (is.null(season_labels)) {
    season_labels <- paste0("S", seq_len(sl))
  }
  result <- data.frame(
    Observation = seq_along(data),
    Season = season_labels[season_index],
    Original = round(data, 2),
    Deseasonalized = round(deseasonalized, 2)
  )
  return(result)
}
