#' @title Seasonalize a Deseasonalized Time Series
#' @param deseasonalized A numeric vector or `ts` object of deseasonalized data.
#' @param indices A numeric vector (or data.frame column) of seasonal indices of length `sl`.
#' @param sl     A positive integer giving the seasonal period.
#' @return A data.frame with columns: Observation (index), Season (label), Deseasonalized (input), Seasonal_Index (from `indices`), Seasonalized (output).
#' @export
#' @examples
#' data <- as.ts(rnorm(48, mean = 100, sd = 48), frequency = 12)
#' sl <- 12
#' si <- SI(data, sl)
#' deseasonalized <- deseasonalize(data, si$Seasonal_Index, sl)
#' seasonalized <- seasonalize(deseasonalized$Deseasonalized, si$Seasonal_Index, sl)
seasonalize <- function(deseasonalized, indices, sl) {
  n <- length(deseasonalized)
  season_index <- rep(seq_len(sl), length.out = n)
  s_index <- as.numeric(indices)
  seasonalized <- deseasonalized * s_index[season_index]
    season_labels <- paste0("S", seq_len(sl))
  result <- data.frame(
    Observation = seq_len(n),
    Season = season_labels[season_index],
    Deseasonalized = deseasonalized,
    Seasonal_Index = s_index[season_index],
    Seasonalized = seasonalized
  )
  return(result)
}
