# minitest - a minimal testing framework v0.0.5 --------------------------------
test_library <- function(package) library(package = package, character.only = TRUE)
test_true <- function(x) invisible(isTRUE(x) || {print(x); stop("the above was returned instead of TRUE")})
test_type <- function(type, x) invisible(typeof(x) == type || {stop("object of type '", typeof(x), "' was returned instead of '", type, "'")})
test_equal <- function(a, b) invisible(a == b || {print(a); print(b); stop("the above expressions were not equal")})
test_identical <- function(a, b) invisible(identical(a, b) || {print(a); print(b); stop("the above expressions were not identical")})
test_error <- function(x, containing = "") invisible(inherits(x <- tryCatch(x, error = identity), "error") && grepl(containing, x[["message"]], fixed = TRUE) || stop("Expected error message containing: ", containing, "\nActual error message: ", x[["message"]]))
# ------------------------------------------------------------------------------

test_library("secretbase")
# Known SHA hashes from NIST:
test_equal(sha3("", 224), "6b4e03423667dbb73b6e15454f0eb1abd4597f9a1b078e3f5b5a6bc7")
test_equal(sha3("", 256), "a7ffc6f8bf1ed76651c14756a061d662f580ff4de43b49fa82d80a4b80f8434a")
test_equal(sha3("", 384), "0c63a75b845e4f7d01107d852e4c2485c51a50aaaa94fc61995e71bbee983a2ac3713831264adb47fb6bd1e058d5f004")
test_equal(sha3("", 512), "a69f73cca23a9ac5c8b567dc185a756e97c982164fe25859e0d1dcc1475c80a615b2123af1f5f94c11e3e9402c3ac558f500199d95b6d3e301758586281dcd26")
test_equal(sha256(""), "e3b0c44298fc1c149afbf4c8996fb92427ae41e4649b934ca495991b7852b855")
# SHA-3 tests:
test_equal(sha3("secret base"), "a721d57570e7ce366adee2fccbe9770723c6e3622549c31c7cab9dbb4a795520")
test_equal(sha3("secret base", bits = 224), "5511b3469d3f1a87b62ce8f0d2dc9510ec5e4547579b8afb32052f99")
test_equal(sha3("secret base", bits = 384L), "79e54f865df004dde10dc2f61baf47eb4637c68d87a2baeb7fe6bc0ac983c2154835ec7deb49b16c246c0dc1d43e32f9")
test_equal(sha3("secret base", bits = "512"), "31076b4690961320a761be0951eeaa9efd0c75c37137a2a50877cbebb8afcc6d7927c41a120ae8fa73fdce8fff726fcbc51d448d020240bc7455963a16e639b1")
test_type("raw", sha3("secret base", convert = FALSE))
# Streaming serialization tests:
test_equal(sha3(data.frame(a = 1, b = 2)), "05d4308e79d029b4af5604739ecc6c4efa1f602a23add0ed2d247b7407d4832f")
test_equal(sha3(c("secret", "base")), "d906024c71828a10e28865a80f5e81d2cb5cd74067d44852d7039813ba62b0b6")
test_equal(sha3(`attr<-`("base", "secret", "base")), "eac181cb1c64e7196c458d40cebfb8bbd6d34a1d728936a2e689465879240e2a")
test_equal(sha3(NULL), "b3e37e4c5def1bfb2841b79ef8503b83d1fed46836b5b913d7c16de92966dcee")
test_equal(sha3(substitute()), "9d31eb41cfb721b8040c52d574df1aacfc381d371c2b933f90792beba5160a57")
# Error handling tests:
test_error(sha3("secret base", bits = 6), "'bits' must be 224, 256, 384 or 512")
test_error(sha3("", convert = 1), "'convert' must be a logical value")
test_error(sha3(file = NULL), "'file' must be a character string")
# File interface tests:
hash_func <- function(file, string) {
  on.exit(unlink(file))
  cat(string, file = file)
  sha3(file = file)
}
test_equal(hash_func(tempfile(), "secret base"), "a721d57570e7ce366adee2fccbe9770723c6e3622549c31c7cab9dbb4a795520")
test_error(hash_func("", ""), "file not found or no read permission")
if (.Platform[["OS.type"]] == "unix") test_error(sha3(file = "~/"), "file read error")
# SHAKE256 tests:
test_equal(shake256("secret base"), "995ebac18dbfeb170606cbbc0f2accce85db4db0dcf4fbe4d3efaf8ccf4e0a94")
test_equal(shake256("secret base", bits = 32, convert = NA), -1044750695L)
test_type("raw", shake256("secret base", convert = FALSE))
test_equal(shake256("secret base", bits = 32), "995ebac1")
test_equal(shake256(shake256("secret base", bits = 32, convert = FALSE), bits = 32), "4d872090")
test_type("character", shake256(rnorm(1e5), bits = 8196))
test_equal(shake256(`class<-`(shake256(character(), bits = 192, convert = FALSE), "hash"), bits = "32", convert = NA), -111175135L)
hash_func <- function(file, string) {
  on.exit(unlink(file))
  cat(string, file = file)
  shake256(file = file)
}
test_equal(hash_func(tempfile(), "secret base"), "995ebac18dbfeb170606cbbc0f2accce85db4db0dcf4fbe4d3efaf8ccf4e0a94")
test_error(shake256("secret base", bits = 0), "'bits' outside valid range of 8 to 2^24")
test_error(shake256("secret base", bits = -1), "'bits' outside valid range of 8 to 2^24")
test_error(shake256("secret base", bits = 2^24 + 1), "'bits' outside valid range of 8 to 2^24")
# Keccak tests:
test_equal(keccak("secret base"), "3fc6092bbec5a434a9933b486a89fa466c1ca013d1e37ab4348ce3764f3463d1")
test_equal(keccak("secret base", bits = 224), "1ddaa7776f138ff5bba898ca7530410a52d09da412c4276bda0682a8")
test_equal(keccak("secret base", bits = 384L), "c82bae24175676028e44aa08b9e2424311847adb0b071c68c7ea47edf049b0e935ddd2fc7c499333bccc08c7eb7b1203")
test_equal(keccak("secret base", bits = "512"), "38297e891d9118e4cf6ff5ba6d6de8c2c3bfa790b425848da7b1d8dffcb4a6a3ca2e32ca0a66f36ce2882786ce2299642de8ffd3bae3b51a1ee145fad555a9d8")
test_type("raw", keccak("secret base", convert = FALSE))
test_error(keccak("secret base", bits = 6), "'bits' must be 224, 256, 384 or 512")
hash_func <- function(file, string) {
  on.exit(unlink(file))
  cat(string, file = file)
  keccak(file = file)
}
test_equal(hash_func(tempfile(), "secret base"), "3fc6092bbec5a434a9933b486a89fa466c1ca013d1e37ab4348ce3764f3463d1")
# SHA-256 tests:
test_equal(sha256("secret base"), "1951c1ca3d50e95e6ede2b1c26fefd0f0e8eba1e51a837f8ccefb583a2b686fe")
test_equal(sha256("secret base", convert = NA)[2L], 1592348733L)
test_equal(sha256(sha256("secret base", convert = FALSE)), "bd45eca9cbd4404cd467909fa8a2196ee9ffc7cb7f70f6343ff6647419744d41")
test_equal(sha256(data.frame(a = 1, b = 2)), "189874c3ac59edecb4eab95a2d7c1bbb293a6ccd04e3da5b28daca91ebc7f15b")
test_equal(sha256(c("secret", "base")), "6a38552b0dab8bf0a6c2f9a6a7acf764631319843f58e85f883301a3cd08b1f2")
test_equal(sha256(`attr<-`("base", "secret", "base")), "6e9e5b1a42304047ba73161360917d001c19974aebd98d345251ff138cddf6ea")
test_equal(sha256(NULL), "71557d1c8bac9bbe3cbec8d00bb223a2f372279827064095447e569fbf5a760a")
test_equal(sha256(substitute()), "4627dde6550f3e355515de5f2c6bb0099161c53c2c7186ce79ab1a74df004bfc")
test_equal(sha256(`class<-`(sha256(character(), convert = FALSE), "hash")), "c762db048ff48f7ba3e9df5539db3aa678cb2f336d96d5c81b4e5d3e19783d14") 
test_error(sha256("", convert = 0), "'convert' must be a logical value")
test_error(sha256(file = NULL), "'file' must be a character string")
hash_func <- function(file, string) {
  on.exit(unlink(file))
  cat(string, file = file)
  sha256(file = file)
}
test_equal(hash_func(tempfile(), "secret base"), "1951c1ca3d50e95e6ede2b1c26fefd0f0e8eba1e51a837f8ccefb583a2b686fe")
test_error(hash_func("", ""), "file not found or no read permission")
if (.Platform[["OS.type"]] == "unix") test_error(sha256(file = "~/"), "file read error")
test_equal(sha256(paste(1:888, collapse = "")), "ec5df945d0ff0c927812ec503fe9ffd5cbdf7cf79b5391ad5002b3a80760183b")
test_equal(sha256("secret", key = "base"), "14b24e4c66bd03c1d6b59bc59e1e47468a437001662ae4be2cb30e0483e13e44")
test_equal(sha256("secret base", key = paste(rep("secret base ", 21L), collapse = "")), "5dab9794515ad176763276bd46f49b029b4578795c52c984243dd636dc0ac11f")
test_equal(sha256("secret base", key = as.raw(1L)), "35a0fc031777e1a16b2c11a614532fbbee5e2ce83271230f62808432a4d13337")
test_equal(sha256("secret base", key = rep(c(as.raw(1L), as.raw(2L)), 64L)), "0d9cbfe4872e0d9ef16f86fbbe5397fd4ed30b7e50b4c5c7722ccf4786aa58d2")
test_equal(sha256("secret base", key = character()), "6bc4693e2025baadf345dd0b133b867ac081dbf6ae02e94e774db4b1a65203ca")
test_error(sha256("secret base", key = list()), "'key' must be a character string, raw vector or NULL")
# SipHash tests:
test_equal(siphash13(""), "2c530c1562a7fbd1")
test_equal(siphash13("", key = ""), "2c530c1562a7fbd1")
test_equal(siphash13("", key = character()), "2c530c1562a7fbd1")
test_equal(siphash13("secret base"), "48c60a316babef0e")
test_equal(siphash13("secret base", key = "secret base"), "2cf27a8f22f02e59")
test_equal(siphash13("secret base", key = c("secret base", "more")), "2cf27a8f22f02e59")
test_equal(siphash13("secret base", key = as.raw(1L)), "5ecd894f7d269521")
test_error(siphash13("", key = list()), "'key' must be a character string, raw vector or NULL") 
test_equal(siphash13("secret base", convert = NA)[2L], 250588011L)
test_equal(siphash13(siphash13("secret base", convert = FALSE)), "498db1332ca02148")
test_equal(siphash13(data.frame(a = 1, b = 2)), "e91a1e412627d654")
test_equal(siphash13(c("secret", "base")), "ddf3c7a54c52150c")
test_equal(siphash13(`attr<-`("base", "secret", "base")), "4a7c3eb69f91f04e")
test_equal(siphash13(NULL), "08d5f59e833de599")
test_equal(siphash13(substitute()), "c8cadc1ab377142a")
test_equal(siphash13(`class<-`(siphash13(character(), convert = FALSE), "hash")), "39124d8b9643418a")
test_error(siphash13("", convert = 0L), "'convert' must be a logical value")
test_error(siphash13(file = NULL), "'file' must be a character string")
hash_func <- function(file, string) {
  on.exit(unlink(file))
  cat(string, file = file)
  siphash13(file = file)
}
test_equal(hash_func(tempfile(), "secret base"), "48c60a316babef0e")
test_error(hash_func("", ""), "file not found or no read permission")
if (.Platform[["OS.type"]] == "unix") test_error(siphash13(file = "~/"), "file read error")
test_equal(siphash13(paste(1:888, collapse = "")), "8337f50b05209c40")
# Base64 tests:
test_type("character", base64enc(c("secret", "base")))
test_type("raw", base64enc(data.frame(), convert = FALSE))
test_type("raw", base64dec(base64enc(as.raw(c(1L, 2L)), convert = FALSE), convert = FALSE))
test_type("integer", base64dec(base64enc(c(1L, 2L)), convert = NA))
test_error(base64dec(base64enc(data.frame())), "data could not be converted to a character string")
test_error(base64enc("", convert = 0), "'convert' must be a logical value")
test_error(base64dec("", convert = 1L), "'convert' must be a logical value")
test_error(base64dec("__"), "input is not valid base64")
test_error(base64dec(404), "input is not valid base64")
test_error(base64dec("c2Vjcm V0IGJhc2U="), "input is not valid base64")
test_error(base64dec("c==="), "input is not valid base64")
test_error(base64dec("c=2="), "input is not valid base64")
test_equal(base64enc(double(1e5), convert = FALSE)[1e5], as.hexmode("41"))
test_equal(base64enc("secret base"), "c2VjcmV0IGJhc2U=")
test_equal(base64dec("c2VjcmV0IGJhc2U= \r\n"), "secret base")
test_equal(base64dec(base64enc("secret base")), "secret base")
test_equal(base64enc("s"), "cw==")
test_equal(base64dec("c2VjcmV0"), "secret")
test_equal(base64enc(""), base64dec(""))
# Base58 tests:
test_type("character", base58enc(sha256("", convert = FALSE)))
test_type("raw", base58enc(data.frame(), convert = FALSE))
test_type("character", base58dec(base58enc("secret base")))
test_type("raw", base58dec(base58enc(as.raw(c(1L, 2L))), convert = FALSE))
test_type("integer", base58dec(base58enc(c(1L, 2L)), convert = NA))
test_equal(base58dec(base58enc("secret base")), "secret base")
test_identical(base58dec(base58enc(sha256("test", convert = FALSE)), convert = FALSE), sha256("test", convert = FALSE))
test_identical(base58dec(base58enc(as.raw(c(1, 2, 3))), convert = FALSE), as.raw(c(1, 2, 3)))
test_identical(base58dec(charToRaw(base58enc("test")), convert = FALSE), charToRaw("test"))
test_identical(base58dec(base58enc(as.raw(c(0, 0, 1))), convert = FALSE), as.raw(c(0, 0, 1)))
test_identical(base58dec(base58enc(raw(0)), convert = FALSE), raw(0))
test_error(base58enc("test", convert = 0), "'convert' must be a logical value")
test_error(base58dec("invalid"), "input is not valid base58")
test_error(base58dec(""), "base58 checksum validation failed")
test_error(base58dec("111"), "base58 checksum validation failed")
test_error(base58dec("1111"), "base58 checksum validation failed")
test_error(base58dec(404), "input is not valid base58")
test_error(base58dec(base58enc("test"), convert = 1L), "'convert' must be a logical value")
test_error(base58dec("\x80"), "input is not valid base58")
# CBOR encoding/decoding tests:
test_type("raw", cborenc(NULL))
test_identical(cbordec(cborenc(NULL)), NULL)
test_equal(cbordec(cborenc(TRUE)), TRUE)
test_equal(cbordec(cborenc(FALSE)), FALSE)
test_identical(cbordec(cborenc(c(TRUE, FALSE, NA))), list(TRUE, FALSE, NA))
test_equal(cbordec(cborenc(3.14)), 3.14)
test_equal(cbordec(cborenc("hello")), "hello")
test_identical(cbordec(cborenc(as.raw(1:10))), as.raw(1:10))
test_identical(cbordec(cborenc(1:3)), as.list(1:3))
test_identical(cbordec(cborenc(c("a", "b"))), as.list(c("a", "b")))
test_identical(cbordec(cborenc(list(1L, "a", TRUE))), list(1L, "a", TRUE))
test_identical(cbordec(cborenc(list(a = 1L, b = "test"))), list(a = 1L, b = "test"))
nested <- list(outer = list(inner = list(value = 42L)), data = as.raw(0:3))
test_identical(cbordec(cborenc(nested)), nested)
test_identical(cbordec(cborenc(raw(0))), raw(0))
test_identical(cbordec(cborenc(list())), list())
test_equal(cbordec(cborenc("")), "")
test_equal(cbordec(cborenc(2147483648)), 2147483648)
test_equal(cbordec(cborenc(-2147483649)), -2147483649)
test_identical(cbordec(cborenc(NA)), NA)
test_identical(cbordec(cborenc(NA_integer_)), NA)
test_identical(cbordec(cborenc(NA_real_)), NA)
test_identical(cbordec(cborenc(NA_character_)), NA)
test_true(is.nan(cbordec(cborenc(NaN))))
test_equal(cbordec(cborenc(Inf)), Inf)
test_equal(cbordec(cborenc(-Inf)), -Inf)
test_equal(cbordec(cborenc("caf\u00e9")), "caf\u00e9")
test_equal(cbordec(cborenc("\U0001F600")), "\U0001F600")
test_equal(cbordec(cborenc(.Machine$integer.max)), .Machine$integer.max)
test_equal(cbordec(cborenc(-.Machine$integer.max)), -.Machine$integer.max)
large <- rep(as.raw(0:255), 20)  # 5120 bytes
test_identical(cbordec(cborenc(large)), large)
test_equal(cbordec(cborenc(255L)), 255L)
test_equal(cbordec(cborenc(256L)), 256L)
test_equal(cbordec(cborenc(65535L)), 65535L)
test_equal(cbordec(cborenc(65536L)), 65536L)
test_equal(cbordec(cborenc(-25L)), -25L)
test_equal(cbordec(cborenc(-257L)), -257L)
test_identical(cbordec(cborenc(c(1L, NA, 3L))), list(1L, NA, 3L))
test_identical(cbordec(cborenc(c(1.5, NA, 2.5))), list(1.5, NA, 2.5))
test_identical(cbordec(cborenc(c("a", NA, "b"))), list("a", NA, "b"))
test_identical(cbordec(as.raw(c(0xfa, 0x41, 0x20, 0x00, 0x00))), 10)
test_identical(cbordec(as.raw(c(0x3b, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00))), -2147483649)
test_identical(cbordec(as.raw(c(0x1b, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00))), 4294967296)  # 64-bit uint (2^32)
test_type("double", cbordec(as.raw(c(0x1a, 0x80, 0x00, 0x00, 0x00))))  # uint > INT_MAX returns double
test_identical(cbordec(as.raw(c(0x01, 0xff))), 1L)  # trailing bytes silently dropped
test_error(cborenc(function() {}), "unsupported type")
test_error(cbordec(raw(0)), "unexpected end")
test_error(cbordec("test"), "must be a raw vector")
test_error(cbordec(as.raw(c(0x19, 0x01))), "unexpected end")  # truncated uint16
test_error(cbordec(as.raw(c(0x1a, 0x01, 0x02))), "unexpected end")  # truncated uint32
test_error(cbordec(as.raw(c(0x1b, 0x01))), "unexpected end")  # truncated uint64
test_error(cbordec(as.raw(0x1c)), "invalid additional info")
test_error(cbordec(as.raw(0xc0)), "unsupported major type")  # tagged item
test_error(cbordec(as.raw(0xe0)), "unsupported simple value")
test_error(cbordec(as.raw(c(0xa1, 0x01, 0x01))), "map key must be text")  # int key
test_error(cbordec(as.raw(c(0x42, 0x01))), "byte string exceeds")  # truncated bytes
test_error(cbordec(as.raw(c(0x62, 0x61))), "text string exceeds")  # truncated text
test_error(cbordec(as.raw(c(0xfb, 0x01))), "float64 exceeds")  # truncated float64
test_error(cbordec(as.raw(c(0xfa, 0x01))), "float32 exceeds")  # truncated float32
test_error(cbordec(as.raw(c(0xf9, 0x01))), "float16 exceeds")  # truncated float16
test_error(cbordec(as.raw(c(0xa1, 0x62, 0x61))), "map key exceeds")  # truncated map key
test_error(cbordec(as.raw(c(rep(0x81, 513), 0x00))), "nesting depth exceeded")  # >512 nested arrays
# RFC 8949 Appendix A test vectors
test_identical(cborenc(0L), as.raw(0x00))
test_identical(cborenc(1L), as.raw(0x01))
test_identical(cborenc(10L), as.raw(0x0a))
test_identical(cborenc(23L), as.raw(0x17))
test_identical(cborenc(24L), as.raw(c(0x18, 0x18)))
test_identical(cborenc(25L), as.raw(c(0x18, 0x19)))
test_identical(cborenc(100L), as.raw(c(0x18, 0x64)))
test_identical(cborenc(1000L), as.raw(c(0x19, 0x03, 0xe8)))
test_identical(cborenc(1000000L), as.raw(c(0x1a, 0x00, 0x0f, 0x42, 0x40)))
test_identical(cborenc(-1L), as.raw(0x20))
test_identical(cborenc(-10L), as.raw(0x29))
test_identical(cborenc(-100L), as.raw(c(0x38, 0x63)))
test_identical(cborenc(-1000L), as.raw(c(0x39, 0x03, 0xe7)))
test_identical(cborenc(raw(0)), as.raw(0x40))
test_identical(cborenc(as.raw(c(0x01, 0x02, 0x03, 0x04))), as.raw(c(0x44, 0x01, 0x02, 0x03, 0x04)))
test_identical(cborenc(""), as.raw(0x60))
test_identical(cborenc("a"), as.raw(c(0x61, 0x61)))
test_identical(cborenc("IETF"), as.raw(c(0x64, 0x49, 0x45, 0x54, 0x46)))
test_identical(cborenc("\"\\"), as.raw(c(0x62, 0x22, 0x5c)))
test_identical(cborenc("\u00fc"), as.raw(c(0x62, 0xc3, 0xbc)))
test_identical(cborenc("\u6c34"), as.raw(c(0x63, 0xe6, 0xb0, 0xb4)))
test_identical(cborenc("\U00010151"), as.raw(c(0x64, 0xf0, 0x90, 0x85, 0x91)))
test_identical(cborenc(list()), as.raw(0x80))
test_identical(cborenc(list(1L, 2L, 3L)), as.raw(c(0x83, 0x01, 0x02, 0x03)))
test_identical(cborenc(list(1L, list(2L, 3L), list(4L, 5L))), as.raw(c(0x83, 0x01, 0x82, 0x02, 0x03, 0x82, 0x04, 0x05)))
test_identical(cborenc(as.list(1:25)), as.raw(c(0x98, 0x19, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x18, 0x18, 0x19)))
test_identical(cborenc(setNames(list(), character(0))), as.raw(0xa0))
test_identical(cborenc(list(a = 1L, b = list(2L, 3L))), as.raw(c(0xa2, 0x61, 0x61, 0x01, 0x61, 0x62, 0x82, 0x02, 0x03)))
test_identical(cborenc(list(a = "A", b = "B", c = "C", d = "D", e = "E")), as.raw(c(0xa5, 0x61, 0x61, 0x61, 0x41, 0x61, 0x62, 0x61, 0x42, 0x61, 0x63, 0x61, 0x43, 0x61, 0x64, 0x61, 0x44, 0x61, 0x65, 0x61, 0x45)))
test_identical(cborenc(FALSE), as.raw(0xf4))
test_identical(cborenc(TRUE), as.raw(0xf5))
test_identical(cborenc(NULL), as.raw(0xf6))
test_identical(cbordec(as.raw(0xf7)), NA)
test_identical(cbordec(as.raw(c(0xfa, 0x47, 0xc3, 0x50, 0x00))), 100000.0)
test_identical(cbordec(as.raw(c(0xfb, 0x3f, 0xf1, 0x99, 0x99, 0x99, 0x99, 0x99, 0x9a))), 1.1)
test_identical(cbordec(as.raw(c(0xfb, 0xc0, 0x10, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66))), -4.1)
test_identical(cbordec(as.raw(c(0xfa, 0x7f, 0x80, 0x00, 0x00))), Inf)
test_identical(cbordec(as.raw(c(0xfb, 0x7f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00))), Inf)
test_identical(cbordec(as.raw(c(0xfa, 0xff, 0x80, 0x00, 0x00))), -Inf)
test_identical(cbordec(as.raw(c(0xfb, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00))), -Inf)
test_true(is.nan(cbordec(as.raw(c(0xfa, 0x7f, 0xc0, 0x00, 0x00)))))
test_true(is.nan(cbordec(as.raw(c(0xfb, 0x7f, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00)))))
test_identical(cbordec(as.raw(c(0xf9, 0x00, 0x00))), 0)
test_identical(cbordec(as.raw(c(0xf9, 0x80, 0x00))), 0)
test_identical(cbordec(as.raw(c(0xf9, 0x3c, 0x00))), 1)
test_identical(cbordec(as.raw(c(0xf9, 0x3e, 0x00))), 1.5)
test_identical(cbordec(as.raw(c(0xf9, 0x7b, 0xff))), 65504)
test_identical(cbordec(as.raw(c(0xf9, 0x04, 0x00))), 0.00006103515625)
test_identical(cbordec(as.raw(c(0xf9, 0xc4, 0x00))), -4)
test_identical(cbordec(as.raw(c(0xf9, 0x7c, 0x00))), Inf)
test_identical(cbordec(as.raw(c(0xf9, 0xfc, 0x00))), -Inf)
test_true(is.nan(cbordec(as.raw(c(0xf9, 0x7e, 0x00)))))
test_identical(cbordec(as.raw(c(0x82, 0x61, 0x61, 0xa1, 0x61, 0x62, 0x61, 0x63))), list("a", list(b = "c")))
