% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sensitivityIxJ.R
\name{sampling.general.sen.IxJ}
\alias{sampling.general.sen.IxJ}
\title{Monte Carlo Sensitivity Analysis for General Permutation-Invariant Test Statistics in I by J Tables}
\usage{
sampling.general.sen.IxJ(
  obs.table,
  gamma,
  delta,
  row = "treatment",
  mc.iteration = 5000,
  transform.fun,
  verbose = FALSE,
  u_space = NULL
)
}
\arguments{
\item{obs.table}{A matrix or table object representing the observed contingency table.}

\item{gamma}{a nonnegative scalar}

\item{delta}{A binary vector with no more than two unique values,
corresponding to treatment levels. The length must match the number of
treatments (rows if \code{row = "treatment"}, columns if \code{row = "outcome"}).}

\item{row}{A string indicating whether rows represent "outcome" or "treatment".
Must be either "outcome" or "treatment". Default is "treatment".}

\item{mc.iteration}{Integer specifying the number of Monte Carlo iterations
for each u-allocation. Higher values increase accuracy but require more
computation time. Default is 5000.}

\item{transform.fun}{A user-defined function that takes a contingency table
and returns a numeric test statistic. This function should be
permutation-invariant within treatment groups.}

\item{verbose}{Logical flag indicating whether to print progress messages
showing current u-allocation and estimated probabilities. Default is FALSE.}

\item{u_space}{Optional matrix where each row represents a candidate u-allocation.
If NULL, a default set of corner allocations is generated. Each row should
have length equal to the number of outcomes.}
}
\value{
A list containing:
\item{rct.prob}{Estimated probability under RCT (all u-allocations zero)}
\item{max.prob}{Maximum estimated probability across all u-allocations}
\item{maximizer}{The u-allocation vector yielding max.prob}
\item{obs.stat}{Observed test statistic value from transform.fun}
\item{obs.table}{The input observed table}
\item{gamma}{Extracted gamma value from the generic bias model}
\item{delta}{The delta vector}
}
\description{
This function implements a sampling-based approach to sensitivity analysis for
general (permutation-invariant) test statistics in I by J contingency tables under
the generic bias model. It uses Sequential Importance Sampling (SIS) from Eisinger and Chen (2017)
to approximate p-values when exact enumeration is computationally infeasible.
}
\details{
This function performs Monte Carlo sensitivity analysis for arbitrary test
statistics that are permutation-invariant. Unlike
the score test version, this function can handle any user-defined test
statistic through the \code{transform.fun} parameter.

The algorithm:
\enumerate{
\item Generates tables using Sequential Importance Sampling (SIS)
\item Evaluates the test statistic on each sampled table
\item Estimates p-values using importance weights
\item Searches over u-allocations to find the maximum p-value
}

When \code{u_space} is not provided, the function generates default corner
allocations that explore extreme cases in the unmeasured confounder space.
}
\examples{
\donttest{
# Example with custom test statistic emphasizing corner cells
obs.table <- matrix(c(10, 5, 8, 12), ncol = 2, byrow = TRUE)

# Define test statistic: sum of diagonal elements
diag_stat <- function(tb) {
  sum(diag(tb))
}

# Run sensitivity analysis
result <- sampling.general.sen.IxJ(
  obs.table = obs.table,
  gamma = 0.5,
  delta = c(0, 1),
  transform.fun = diag_stat,
  mc.iteration = 5000,
  verbose = TRUE
)

# Custom u-space example
custom_u <- matrix(c(0, 0,
                    5, 0,
                    0, 8,
                    5, 8), ncol = 2, byrow = TRUE)

result_custom <- sampling.general.sen.IxJ(
  obs.table = obs.table,
  gamma = 0.5,
  delta = c(0, 1),
  transform.fun = diag_stat,
  u_space = custom_u
)
}

}
\references{
Eisinger, R. D., & Chen, Y. (2017). Sampling for Conditional Inference on
Contingency Tables. Journal of Computational and Graphical Statistics, 26(1),
79–87.
}
\seealso{
\code{\link{exact.general.sen.IxJ}} for exact computation when feasible,
\code{\link{sampling.score.sen.IxJ}} for score tests with ordinal data,
}
