% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/task_generic_indic.R
\name{generic_sews}
\alias{generic_sews}
\title{Generic Spatial Early-Warning signals}
\usage{
generic_sews(
  mat,
  subsize = 4,
  abs_skewness = FALSE,
  moranI_coarse_grain = FALSE
)
}
\arguments{
\item{mat}{A matrix (quantitative data), a binary matrix (which contains 
\code{TRUE} or \code{FALSE} values) 
or a list of those}

\item{subsize}{The subsize used for the coarse-graining phase (see Details)}

\item{abs_skewness}{Should the absolute skewness be used instead of its 
raw values ?}

\item{moranI_coarse_grain}{Should the input matrix be coarse-grained before
computing the Moran's I indicator value ?}
}
\value{
\code{generic_sews} returns an object of class \code{simple_sews_single}
  (a list) if \code{mat} is a single matrix or an object of class 
  \code{simple_sews_list} if \code{mat} is a list. You probably want to use some  
  of the methods written for these complicated objects instead of extracting 
  values directly (they are displayed using \code{print(<object>)}).

\code{indictest} returns an object of class \code{generic_test} (a \code{data.frame}). 

\code{plot} methods return \code{ggplot} objects, usually immediately displayed 
  when \code{plot} is being called at the R prompt.
}
\description{
Computation, significance assessment and display of spatial 
  generic early warning signals (Moran's I, variance and skewness)
}
\details{
The Generic Early warning signal are based on the property of a 
  dynamical system to "slow down" when approaching a critical point, 
  that is, take more time to return to equilibrium after a perturbation. This 
  is expected to be reflected in several spatial characteristics: the 
  variance, the spatial autocorrelation (at lag-1) and the skewness. This 
  function provides a convenient workflow to compute these indicators, 
  assess their significance and display the results. 

Before computing the actual indicators, the matrix can be "coarse-grained". 
  This process reduces the matrix by averaging the nearby cells using 
  a square window defined by the \code{subsize} parameter. This makes spatial  
  variance and skewness reflect actual spatial patterns when working with 
  binary (\code{TRUE}/\code{FALSE} data), but is optional when using 
  continuous data. Keep in mind that it effectively reduces the size of 
  the matrix by approximately \code{subsize} on each dimension. 
  
The significance of generic early-warning signals can be estimated by 
  reshuffling the original matrix (function \code{indictest}). Indicators 
  are then recomputed on the shuffled matrices and the values obtained are 
  used as a null distribution. P-values are obtained based on the rank of 
  the observed value in the null distribution. A small P-value means 
  that the indicator is significantly above the null values, as expected 
  before a critical point. 

The \code{plot} method can displays the results graphically. A text summary 
  can be obtained using the \code{summary} method.
}
\examples{

data(serengeti)
gen_indic <- generic_sews(serengeti, subsize = 5, 
                           moranI_coarse_grain = TRUE)

# Display results
summary(gen_indic)

# Display trends along the varying model parameter
plot(gen_indic, along = serengeti.rain)

# Compute significance (long)
\donttest{
gen_test <- indictest(gen_indic, nulln = 199)

print(gen_test)

# Display the trend, now with a grey ribbon indicating the 5\%-95\% quantile
# range of the null distribution
plot(gen_test, along = serengeti.rain)

# Display the effect size compared to null distribution 
plot(gen_test, along = serengeti.rain, what = "z_score")

# Note that plot() method returns a ggplot object that can be modified
# for convenience
if ( require(ggplot2) ) { 
  plot(gen_test, along = serengeti.rain) + 
    geom_vline(xintercept = 733, color = "red", linetype = "dashed") +
    xlab('Annual rainfall') + 
    theme_minimal()
}

}

}
\references{
Kefi, S., Guttal, V., Brock, W.A., Carpenter, S.R., Ellison, A.M., 
  Livina, V.N., et al. (2014). Early Warning Signals of Ecological 
  Transitions: Methods for Spatial Patterns. PLoS ONE, 9, e92097.
  
  Dakos, V., van Nes, E. H., Donangelo, R., Fort, H., & 
  Scheffer, M. (2010). Spatial correlation as leading indicator of 
  catastrophic shifts. Theoretical Ecology, 3(3), 163-174.
  
  Guttal, V., & Jayaprakash, C. (2008). Spatial variance and spatial 
  skewness: leading indicators of regime shifts in spatial ecological 
  systems. Theoretical Ecology, 2(1), 3-12.
}
\seealso{
\code{\link{indictest}}, to test the significance of indicator values. 
  Individual indicators: \code{\link{raw_cg_moran}}
    \code{\link{raw_cg_variance}}, \code{\link{raw_cg_skewness}}, 
    \code{\link{simple_sews}}
}
