% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/generics.r
\name{plot.specr.object}
\alias{plot.specr.object}
\title{Plot specification curve and analytic choices}
\usage{
\method{plot}{specr.object}(
  x,
  type = "default",
  var = .data$estimate,
  group = NULL,
  choices = c("x", "y", "model", "controls", "subsets"),
  labels = c("A", "B"),
  rel_heights = c(2, 3),
  desc = FALSE,
  null = 0,
  ci = TRUE,
  ribbon = FALSE,
  formula = NULL,
  print = TRUE,
  ...
)
}
\arguments{
\item{x}{A \code{specr.object} object, usually resulting from calling \code{specr()}.}

\item{type}{What type of figure should be plotted? If \code{type = "default"},
the standard specification curve analysis plot (the specification curve as the
upper panel and an overview of the relevant choices as the lower panel) is
created. If \code{type = "curve"}, only the specification curve (upper panel
of the default plot) is plotted. If \code{type = "choices"}, only the choice
panel (lower part of the default plot) is plotted. If \code{type = "boxplot"},
an alternative visualization of differences between choices is plotted that
summarizes results per choice using box-and-whisker plot(s). If
\code{type = "samplesizes"}, a barplot of sample sizes per specification is
plotted.  See examples for more information.}

\item{var}{Which parameter should be plotted in the curve? Defaults to
\code{estimate}, but other parameters (e.g., p.value, fit_r.squared,...)
can be plotted too.}

\item{group}{Should the arrangement of the curve be grouped by a particular choice?
Defaults to NULL, but can be any of the present choices (e.g., x, y, controls...)}

\item{choices}{A vector specifying which analytic choices should be plotted.
By default, all choices (x, y, model, controls, subsets) are plotted.}

\item{labels}{Labels for the two parts of the plot}

\item{rel_heights}{vector indicating the relative heights of the plot.}

\item{desc}{Logical value indicating whether the curve should the arranged in
a descending order. Defaults to FALSE.}

\item{null}{Indicate what value represents the 'null' hypothesis (defaults to
zero).}

\item{ci}{Logical value indicating whether confidence intervals should be
plotted.}

\item{ribbon}{Logical value indicating whether a ribbon instead should be
plotted}

\item{formula}{In combination with \code{type = "variance"}, you can provide
a specific formula to extract specific variance components. The syntax of the
formula is based on \code{lme4::lmer()} and thus looks something like, e.g.:
\code{"estimate ~ 1 + (1|x) + (1|y)"} (to estimate the amount of variance
explained by different independent \code{x} and dependent variables \code{y}). All other
choices are then subsumed under residual variance. By no formula is provided,
all choices (x, y, model, controls, and subsets) that have more than one alternative
are included. See examples for further details.}

\item{print}{In combination with \code{type = "variance"}, logical value indicating
whether the intra-class correlations (i.e., percentages of variance explained by
analstical choices) should be printed or not. Defaults to TRUE.}

\item{...}{further arguments passed to or from other methods (currently ignored).}
}
\value{
A \link[ggplot2]{ggplot} object that can be customized further.
}
\description{
This function plots visualizations of the specification curve
analysis. The function requires an object of class \code{specr.object}, usually
the results of calling \code{specr()} to create a standard visualization of the
specification curve analysis. Several types of visualizations are possible.
}
\examples{
\dontrun{
# Specification Curve analysis ----
# Setup specifications
specs <- setup(data = example_data,
   y = c("y1", "y2"),
   x = c("x1", "x2"),
   model = "lm",
   controls = c("c1", "c2"),
   subsets = list(group1 = unique(example_data$group1),
                  group2 = unique(example_data$group2)))

# Run analysis
results <- specr(specs)

# Resulting data frame with estimates
as_tibble(results)  # This will be used for plotting


# Visualizations ---
# Plot results in various ways
plot(results)                            # default
plot(results, choices = c("x", "y"))     # specific choices
plot(results, ci = FALSE, ribbon = TRUE) # exclude CI and add ribbon instead
plot(results, type = "curve")
plot(results, type = "choices")
plot(results, type = "samplesizes")
plot(results, type = "boxplot")


# Grouped plot
plot(results, group = controls)

# Alternative and specific visualizations ----
# Other variables in the resulting data set can be plotted too
plot(results,
     type = "curve",
     var = fit_r.squared,   # extract "r-square" instead of "estimate"
     ci = FALSE)

# Such a plot can also be extended (e.g., by again adding the estimates with
# confidence intervals)
library(ggplot2)
plot(results, type = "curve", var = fit_r.squared) +
  geom_point(aes(y = estimate), shape = 5) +
  labs(x = "specifications", y = "r-squared | estimate")

# We can also investigate how much variance is explained by each analytical choice
plot(results, type = "variance")

# By providing a specific formula in `lme4::lmer()`-style, we can extract specific choices
# and also include interactions between chocies
plot(results,
     type = "variance",
     formula = "estimate ~ 1 + (1|x) + (1|y) + (1|group1) + (1|x:y)")

## Combining several plots ----
# `specr` also exports the function `plot_grid()` from the package `cowplot`, which
# can be used to combine plots meaningfully
a <- plot(results, "curve")
b <- plot(results, "choices", choices = c("x", "y", "controls"))
c <- plot(results, "samplesizes")
plot_grid(a, b, c,
          align = "v",
          axis = "rbl",
          rel_heights = c(2, 3, 1),
          ncol = 1)
}
}
