% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cspgram.R, R/fftpgram.R, R/lspgram.R,
%   R/spectr.R
\name{cspgram}
\alias{cspgram}
\alias{fftpgram}
\alias{lspgram}
\alias{spectr}
\title{Calculate periodogram}
\usage{
cspgram(
  x,
  deltat,
  periodRange = c(18, 32),
  method = c("greedy", "conservative", "standard"),
  na.action = stats::na.fail,
  dopar = FALSE
)

fftpgram(
  x,
  deltat,
  periodRange = c(18, 32),
  pad = 50,
  na.action = stats::na.fail,
  ...
)

lspgram(x, deltat, time, periodRange = c(18, 32), ofac = 50)

spectr(
  x,
  deltat,
  time,
  periodRange = c(18, 32),
  method = c("greedy_chisq", "conservative_chisq", "standard_chisq", "lombscargle",
    "fft"),
  ofac = 50,
  pad = 50,
  na.action = stats::na.fail,
  dopar = FALSE,
  ...
)
}
\arguments{
\item{x}{Numeric vector of measurements.}

\item{deltat}{Numeric value of the interval between time-points.}

\item{periodRange}{Numeric vector of the minimum and maximum values of the
period to consider, in the same units as \code{deltat} or \code{time}.}

\item{method}{Character indicating which method to use. Can be an unambiguous
substring of the full name.}

\item{na.action}{Function specifying how to handle \code{NA} values in \code{x}.
Default is \code{\link[stats:na.fail]{stats::na.fail()}}, which gives an error if any values are
missing. Ignored for Lomb-Scargle.}

\item{dopar}{Logical indicating whether to run calculations in parallel if
a parallel backend is already set up, e.g., using
\code{\link[doParallel:registerDoParallel]{doParallel::registerDoParallel()}}. Only used for chi-square.}

\item{pad}{Numeric value of the proportion of the length of \code{x} by which to
pad \code{x} with zeros. Must be > 0. Only used for FFT.}

\item{...}{Other arguments passed to \code{\link[stats:spec.pgram]{stats::spec.pgram()}} for FFT.}

\item{time}{Numeric vector of time-points. Can be specified instead of
\code{deltat} for Lomb-Scargle.}

\item{ofac}{Integer value of the oversampling factor. Must be >= 1. Only used
for Lomb-Scargle.}
}
\value{
A \code{data.table} with various columns depending on the method. For any
version of chi-square, columns will be \code{period}, \code{chisq}, \code{df}, and
\code{log_pval}. The log p-value is more reliable than the p-value, since R has
finite precision, so p-values less than about 5e-324 would be set to 0. For
Lomb-Scargle and FFT, columns will be \code{period} and \code{power}.
}
\description{
Calculate periodogram for a time-course using Lomb-Scargle, fast Fourier
transform, or selected version of chi-square. The \code{spectr} function is a
wrapper for the various methods. \code{lspgram} is in turn a wrapper for
\code{\link[lomb:lsp]{lomb::lsp()}}, and \code{fftpgram} a wrapper for \code{\link[stats:spec.pgram]{stats::spec.pgram()}}. Among the
versions of chi-square, it is highly recommended to use greedy, which has
lower bias than standard and lower variance than conservative.
}
\examples{
library('data.table')

set.seed(1789)
deltat = 0.1
tau = 25
tt = seq(0, 24 * 3, deltat)
x = 3 * sin(tt / tau * 2 * pi) + rnorm(length(tt))

specCsp = spectr(x, deltat, method = 'greedy')
peakCsp = specCsp[which.min(log_pval)]

specLsp = spectr(x, deltat, method = 'lomb')
peakLsp = specLsp[which.max(power)]

specFft = spectr(x, deltat, method = 'fft')
peakFft = specFft[which.max(power)]

}
