#' Graph Information Criterion (GIC)
#'
#' \code{GIC} returns the Kullback-Leibler divergence, L1 or L2 distance between an
#' undirected graph and a given graph model using the exact or degree-based spectral densities.
#'
#' @param Graph the undirected graph (igraph object).
#' If \code{Graph} has the  attribute \code{eigenvalues} containing
#' the eigenvalues of \code{Graph}, such values will be used to
#' compute its spectral density.
#'
#' @param model either a list, a string, or a function describing a
#' graph model:
#'
#' A list that represents the spectral density of a model. It contains the
#' components 'x' and 'y', which are numeric vectors of the same size. The 'x'
#' component contains the points at which the density was computed and the 'y'
#' component contains the observed density.
#'
#' A string that indicates one of the following models: 'ER' (Erdos-Renyi random
#' graph), 'GRG' (geometric random graph), 'KR' (k regular random graph), 'WS'
#' (Watts-Strogatz model), and 'BA' (Barabási-Albert model). When the argument
#' \code{model} is a string, the user must also provide the model parameter by using
#' the argument \code{p}.
#'
#' A function that returns a graph (igraph object)
#' generated by a graph model. It must contain two arguments: the first one
#' corresponds to the graph size and the second to the parameter of the model.
#' The model parameter will be provided by the argument \code{p} of the \code{GIC}
#' function.
#'
#' @param p the model parameter. The user must provide a valid parameter if the
#' argument \code{model} is a string or a function.
#' For the predefined models ('ER', 'GRG', 'KR', 'WS', and 'BA'), the parameter
# \code{p} corresponds to:
#' the probability to connect a pair of vertices, for the 'ER' model
#' (Erdos-Renyi random graph);
#'
#' the radius used to construct the geometric graph in a unit square, for the
#' 'GRG' model (geometric random graph);
#'
#' the degree \code{k} of a regular graph, for the 'KR' model (k regular random
#' graph);
#'
#' the probability to reconnect a vertex, for the 'WS' model (Watts-Strogatz
#' model);
#'
#' and the scaling exponent, for the 'BA' model (Barabási-Albert model).
#'
#' @param dist string indicating if you want to use the 'KL' (default), 'L1' or 'L2'
#' distances. 'KL' means Kullback-Leibler divergence.
#'
#' @param ... Other relevant parameters for \code{\link{graph.spectral.density}}.
#'
#' @return A list with class 'statGraph' containing the following components:
#' \item{\code{method:}}{ a string indicating the used method.}
#' \item{\code{info:}}{ a string showing details about the method.}
#' \item{\code{data.name:}}{ a string with the data's name(s).}
#' \item{\code{value:}}{ a real number corresponding to the Kullback-Leibler divergence, L1, or
#'  L2 distance between the observed graph and the graph model.}
#'
#' @keywords graph_information_criterion
#'
#' @references
#' Takahashi, D. Y., Sato, J. R., Ferreira, C. E. and Fujita A. (2012)
#' Discriminating Different Classes of Biological Networks by Analyzing the
#' Graph Spectra  Distribution. _PLoS ONE_, *7*, e49949.
#' doi:10.1371/journal.pone.0049949.
#'
#' Silverman, B. W. (1986) _Density Estimation_.  London: Chapman and Hall.
#'
#' Sturges, H. A. The Choice of a Class Interval. _J. Am. Statist. Assoc._,
#' *21*, 65-66.
#'
#' Sheather, S. J. and Jones, M. C. (1991). A reliable data-based bandwidth
#' selection method for kernel density estimation.
#' _Journal of the Royal Statistical Society series B_, 53, 683-690.
#' http://www.jstor.org/stable/2345597.
#'
#' @examples
#' set.seed(1)
#' G <- igraph::sample_gnp(n=50, p=0.5)
#'
#' # Using a string to indicate the graph model
#' result1 <- GIC(G, 'ER', 0.5)
#' result1
#'
#' # Using a function to describe the graph model
#' # Erdos-Renyi graph
#' model <- function(n, p) {
#'    return (igraph::sample_gnp(n, p))
#' }
#' result2 <- GIC(G, model, 0.5)
#' result2
#'
#' @import methods
#' @export
GIC <- function(Graph, model, p = NULL, dist = "KL", ...) {
    if (!valid.input(Graph)) {
        stop("The input should be an igraph object!")
    }
    data.name <- deparse(substitute(Graph))

    n <- igraph::vcount(Graph)
    m <- igraph::ecount(Graph)
    avg_deg <- as.integer(ceiling(m/n))

    graph_den <- Graph$density
    model_den <- NULL
    if (is.null(graph_den)) {
        model_den <- graph.model.spectral.density(model = model, n = n, p = p, mean_deg = avg_deg, ...)
        graph_den <- graph.spectral.density(Graph, from = model_den$from, to = model_den$to, ...)
    } else {
        model_den <- graph.model.spectral.density(model, n = n, p = p, mean_deg = avg_deg, from = graph_den$from, to = graph_den$to, ...)
    }
    #
    if (sum(is.na(model_den)) > 0)
        return(Inf)
    if (sum(is.na(graph_den)) > 0)
        return(Inf)
    #
    out <- distance(graph_den, model_den, dist = dist)
    ###
    method_info <- "Graph Information Criterion"
    output <- list(method = method_info, info = graph_den$info, data.name = data.name, value = out, dist = dist)
    class(output) <- "statGraph"
    return(output)
}
