#' Principal Component Analysis (PCA)
#'
#' Performs a principal component analysis on a numeric matrix or data frame.
#' Optionally centers and scales the variables before the analysis.
#'
#' @param X Numeric matrix or data frame with observations in rows and variables in columns.
#' @param reduce Logical. If TRUE (default), each variable is centered and scaled before the analysis.
#'
#' @return A list with the following elements:
#' \item{U}{Vector of eigenvalues.}
#' \item{V}{Matrix whose columns are the eigenvectors (principal components).}
#' \item{C}{Component scores matrix (data projection).}
#' \item{CP}{Matrix of principal correlations.}
#' \item{Q}{Matrix with the percentage of representation per individual (per row).}
#' \item{R}{Correlation matrix if \code{reduce=TRUE}.}
#'
#' @noRd
#' @keywords internal
acp <- function(X, reduce = TRUE) {
  # ----------------------------------------------------------------------
  # Basic validation
  # - The function accepts either a matrix or a data frame.
  # - Everything is converted to a numeric matrix.
  # - At least 2 rows (individuals) and 1 column (variable) are required.
  if (!is.matrix(X) && !is.data.frame(X)) stop("X must be numeric matrix/data.frame.")
  X <- as.matrix(X)
  if (!is.numeric(X)) stop("X must be numeric.")
  n <- nrow(X); m <- ncol(X)
  if (n < 2 || m < 1) stop("X must have >= 2 rows and >= 1 column.")

  # ----------------------------------------------------------------------
  # Step 1: Center and Reduce (optional)
  if (reduce) {
    if (!exists("center.and.reduce")) stop("Define this function first: center.and.reduce()")
    Xc <- center.and.reduce(X)
  } else {
    Xc <- X
  }

  # ----------------------------------------------------------------------
  # Step 2: matrix R = (1/n) Xc'Xc (symmetrized for numerical stability)
  R <- (1 / n) * crossprod(Xc)
  R <- (R + t(R)) / 2

  # ----------------------------------------------------------------------
  # Step 3: eigenvalues (there may be small negative values due to rounding)
  ee <- eigen(R, symmetric = TRUE)
  U <- as.numeric(ee$values)
  V <- ee$vectors
  Upos <- pmax(U, 0)  # clamp to 0 to avoid NaN in sqrt

  # ----------------------------------------------------------------------
  # Step 4: Principal components (scores)
  C <- Xc %*% V

  # ----------------------------------------------------------------------
  # Step 5: (Optional) Representation quality (Q matrix, % representation per individual)
  denom <- rowSums(X^2)
  denom[denom == 0] <- NA_real_
  Q <- sweep(C^2, 1, denom, "/")

  # ----------------------------------------------------------------------
  # Step 6: Main correlations (loadings)
  CP <- sweep(V, 2, sqrt(Upos), `*`)

  list(U = U, V = V, C = C, CP = CP, Q = Q, R = R)
}
