% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/quadratic_forms.R
\name{make_quad_form_matrix}
\alias{make_quad_form_matrix}
\title{Get Variance Estimator's Quadratic Form Matrix}
\usage{
make_quad_form_matrix(
  variance_estimator = "Yates-Grundy",
  probs = NULL,
  joint_probs = NULL,
  cluster_ids = NULL,
  strata_ids = NULL,
  strata_pop_sizes = NULL,
  sort_order = NULL,
  aux_vars = NULL
)
}
\arguments{
\item{variance_estimator}{The name of the variance estimator
whose quadratic form matrix should be created. See the section "Variance Estimators" below.
Options include:
\itemize{
\item \strong{"Yates-Grundy"}: The Yates-Grundy variance estimator based on
first-order and second-order inclusion probabilities. If this is used,
the argument \code{joint_probs} must also be used.
\item \strong{"Horvitz-Thompson"}: The Horvitz-Thompson variance estimator based on
first-order and second-order inclusion probabilities. If this is used,
the argument \code{joint_probs} must also be used.
\item \strong{"Stratified Multistage SRS"}: The usual stratified multistage variance estimator
based on estimating the variance of cluster totals within strata at each stage.
If this option is used, then it is necessary to also use the arguments
\code{strata_ids}, \code{cluster_ids}, \code{strata_pop_sizes}, and \code{strata_pop_sizes}.
\item \strong{"Ultimate Cluster"}: The usual variance estimator based on estimating
the variance of first-stage cluster totals within first-stage strata.
If this option is used, then it is necessary to also use the arguments
\code{strata_ids}, \code{cluster_ids}, \code{strata_pop_sizes}.
Optionally, to use finite population correction factors, one can also use the argument \code{strata_pop_sizes}.
\item \strong{"Deville-1"}: A variance estimator for unequal-probability
sampling without replacement, described in Matei and Tillé (2005)
as "Deville 1". If this option is used, then it is necessary to also use the arguments
\code{strata_ids}, \code{cluster_ids}, and \code{probs}.
\item \strong{"Deville-2"}:  A variance estimator for unequal-probability
sampling without replacement, described in Matei and Tillé (2005)
as "Deville 2". If this option is used, then it is necessary to also use the arguments
\code{strata_ids}, \code{cluster_ids}, and \code{probs}.
\item \strong{"SD1"}: The non-circular successive-differences variance estimator described by Ash (2014),
sometimes used for variance estimation for systematic sampling.
\item \strong{"SD2"}: The circular successive-differences variance estimator described by Ash (2014).
This estimator is the basis of the "successive-differences replication" estimator commonly used
for variance estimation for systematic sampling.
\item \strong{"BOSB"}: \cr The kernel-based variance estimator proposed by
Breidt, Opsomer, and Sanchez-Borrego (2016) for use with systematic samples
or other finely stratified designs. Uses the Epanechnikov kernel
with the bandwidth automatically chosen to result in the smallest possible
nonempty kernel window.
\item \strong{"Deville-Tille"}: The estimator of Deville and Tillé (2005),
developed for balanced sampling using the cube method.
\item \strong{"Beaumont-Emond"}: The variance estimator of Beaumont and Emond (2022)
for multistage unequal-probability sampling without replacement.
}}

\item{probs}{Required if \code{variance_estimator} equals \code{"Deville-1"}, \code{"Deville-2"}, or \code{"Deville-Tille"}.
This should be a matrix or data frame of sampling probabilities.
If there are multiple stages of sampling,
then \code{probs} can have multiple columns,
with one column for each level of sampling to be accounted for by the variance estimator.}

\item{joint_probs}{Only used if \code{variance_estimator = "Horvitz-Thompson"} or \code{variance_estimator = "Yates-Grundy"}.
This should be a matrix of joint inclusion probabilities.
Element \code{[i,i]} of the matrix is the first-order inclusion probability of unit \code{i},
while element \code{[i,j]} is the joint inclusion probability of units \code{i} and \code{j}.}

\item{cluster_ids}{Required unless \code{variance_estimator} equals \code{"Horvitz-Thompson"} or \code{"Yates-Grundy"}.
This should be a matrix or data frame of cluster IDs. If there are multiple stages of sampling,
then \code{cluster_ids} can have multiple columns,
with one column for each level of sampling to be accounted for by the variance estimator.}

\item{strata_ids}{Required if \code{variance_estimator} equals \code{"Stratified Multistage SRS"}
or \code{"Ultimate Cluster"}.
This should be a matrix or data frame of strata IDs. If there are multiple stages of sampling,
then \code{strata_ids} can have multiple columns,
with one column for each level of sampling to be accounted for by the variance estimator.}

\item{strata_pop_sizes}{Required if \code{variance_estimator} equals \code{"Stratified Multistage SRS"},
but can optionally be used if \code{variance_estimator} equals \code{"Ultimate Cluster"}, \code{"SD1"}, or \code{"SD2"}.
If there are multiple stages of sampling,
then \code{strata_pop_sizes} can have multiple columns,
with one column for each level of sampling to be accounted for by the variance estimator.}

\item{sort_order}{Required if \code{variance_estimator} equals \code{"SD1"} or \code{"SD2"}.
This should be a vector that orders the rows of data into the order used for sampling.}

\item{aux_vars}{Required if \code{variance_estimator} equals \code{"Deville-Tille"}.
A matrix of auxiliary variables.}
}
\value{
The matrix of the quadratic form representing the variance estimator.
}
\description{
Common variance estimators for estimated population totals can be represented as a quadratic form.
Given a choice of variance estimator and information about the sample design,
this function constructs the matrix of the quadratic form.
\cr \cr
In notation, let
\eqn{v(\hat{Y}) = \mathbf{\breve{y}}^{\prime}\mathbf{\Sigma}\mathbf{\breve{y}}},
where \eqn{\breve{y}} is the vector of weighted values, \eqn{y_i/\pi_i, \space i=1,\dots,n}.
This function constructs the \eqn{n \times n} matrix of the quadratic form, \eqn{\mathbf{\Sigma}}.
}
\section{Variance Estimators}{

See \link[svrep]{variance-estimators} for a
description of each variance estimator.
}

\section{Arguments required for each variance estimator}{

Below are the arguments that are required or optional for each variance estimator.\tabular{lrrrrrrr}{
   variance_estimator \tab probs \tab joint_probs \tab cluster_ids \tab strata_ids \tab strata_pop_sizes \tab sort_order \tab aux_vars \cr
   Stratified Multistage SRS \tab  \tab  \tab Required \tab Required \tab Required \tab  \tab  \cr
   Ultimate Cluster \tab  \tab  \tab Required \tab Required \tab Optional \tab  \tab  \cr
   SD1 \tab  \tab  \tab Required \tab Optional \tab Optional \tab Required \tab  \cr
   SD2 \tab  \tab  \tab Required \tab Optional \tab Optional \tab Required \tab  \cr
   Deville-1 \tab Required \tab  \tab Required \tab Optional \tab  \tab  \tab  \cr
   Deville-2 \tab Required \tab  \tab Required \tab Optional \tab  \tab  \tab  \cr
   Beaumont-Emond \tab Required \tab  \tab Required \tab Optional \tab  \tab  \tab  \cr
   Deville-Tille \tab Required \tab  \tab Required \tab Optional \tab  \tab  \tab Required \cr
   BOSB \tab  \tab  \tab Required \tab Optional \tab  \tab  \tab Required \cr
   Yates-Grundy \tab  \tab Required \tab  \tab  \tab  \tab  \tab  \cr
   Horvitz-Thompson \tab  \tab Required \tab  \tab  \tab  \tab  \tab  \cr
}
}

\examples{
\donttest{
# Example 1: The Horvitz-Thompson Estimator
  library(survey)
  data("election", package = "survey")

  ht_quad_form_matrix <- make_quad_form_matrix(variance_estimator = "Horvitz-Thompson",
                                               joint_probs = election_jointprob)
  ##_ Produce variance estimate
  wtd_y <- as.matrix(election_pps$wt * election_pps$Bush)
  t(wtd_y) \%*\% ht_quad_form_matrix \%*\% wtd_y

  ##_ Compare against result from 'survey' package
  svytotal(x = ~ Bush,
           design = svydesign(data=election_pps,
                              variance = "HT",
                              pps = ppsmat(election_jointprob),
                              ids = ~ 1, fpc = ~ p)) |> vcov()

# Example 2: Stratified multistage Sample ----

  data("mu284", package = 'survey')
  multistage_srswor_design <- svydesign(data = mu284,
                                        ids = ~ id1 + id2,
                                        fpc = ~ n1 + n2)

  multistage_srs_quad_form <- make_quad_form_matrix(
    variance_estimator = "Stratified Multistage SRS",
    cluster_ids = mu284[,c('id1', 'id2')],
    strata_ids = matrix(1, nrow = nrow(mu284), ncol = 2),
    strata_pop_sizes = mu284[,c('n1', 'n2')]
  )

  wtd_y <- as.matrix(weights(multistage_srswor_design) * mu284$y1)
  t(wtd_y) \%*\% multistage_srs_quad_form \%*\% wtd_y

  ##_ Compare against result from 'survey' package
  svytotal(x = ~ y1, design = multistage_srswor_design) |> vcov()

# Example 3: Successive-differences estimator ----

  data('library_stsys_sample', package = 'svrep')

  sd1_quad_form <- make_quad_form_matrix(
    variance_estimator = 'SD1',
    cluster_ids = library_stsys_sample[,'FSCSKEY',drop=FALSE],
    strata_ids = library_stsys_sample[,'SAMPLING_STRATUM',drop=FALSE],
    strata_pop_sizes = library_stsys_sample[,'STRATUM_POP_SIZE',drop=FALSE],
    sort_order = library_stsys_sample[['SAMPLING_SORT_ORDER']]
  )

  wtd_y <- as.matrix(library_stsys_sample[['TOTCIR']] /
                      library_stsys_sample$SAMPLING_PROB)
  wtd_y[is.na(wtd_y)] <- 0

  t(wtd_y) \%*\% sd1_quad_form \%*\% wtd_y

# Example 4: Deville estimators ----

 data('library_multistage_sample', package = 'svrep')

 deville_quad_form <- make_quad_form_matrix(
     variance_estimator = 'Deville-1',
     cluster_ids = library_multistage_sample[,c("PSU_ID", "SSU_ID")],
     strata_ids = cbind(rep(1, times = nrow(library_multistage_sample)),
                        library_multistage_sample$PSU_ID),
     probs = library_multistage_sample[,c("PSU_SAMPLING_PROB",
                                          "SSU_SAMPLING_PROB")]
 )
}
}
\seealso{
See \link[svrep]{variance-estimators} for a
description of each variance estimator.

For a two-phase design, the function
\link[svrep]{make_twophase_quad_form} combines
the quadratic form matrix from each phase.
}
