% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/final_prop_svyglm.R
\name{final_prop_svyglm}
\alias{final_prop_svyglm}
\title{Propensity-weighted survey GLM}
\usage{
final_prop_svyglm(
  data,
  dep_var,
  covariates,
  exposure,
  id_var,
  strata_var,
  weight_var,
  exposure_type = "binary",
  outcome_covariates = NULL,
  level = 0.95,
  ...
)
}
\arguments{
\item{data}{Data frame}

\item{dep_var}{Character; binary outcome}

\item{covariates}{Character vector; adjustment variables}

\item{exposure}{Character; treatment/exposure variable}

\item{id_var}{Character; PSU}

\item{strata_var}{Character; strata}

\item{weight_var}{Character; base weight}

\item{exposure_type}{Character; "binary", "multinomial", "continuous"}

\item{outcome_covariates}{Character vector of additional covariates to include in the final outcome model after applying propensity weights (default = NULL)}

\item{level}{Numeric; confidence interval level}

\item{...}{Additional args to svyglm}
}
\value{
A list with:
\itemize{
\item \code{ps_model}: Propensity score svyglm model.
\item \code{final_model}: Weighted outcome svyglm model.
\item \code{OR_table}: Odds ratios with CI and p-values.
\item \code{AUC}: Weighted AUC.
\item \code{data}: Data with IPTW and predictions.
}
}
\description{
Calculates IPTW weights and fits survey-weighted GLM.
Supports binary, multinomial, or continuous exposures.
}
\examples{
set.seed(123)
n <- 1500
dat <- data.frame(
  psu = sample(1:10, n, replace = TRUE),
  strata = sample(1:5, n, replace = TRUE),
  weight = runif(n, 0.5, 2),
  age = rnorm(n, 50, 10),
  sex = factor(sample(c("Male", "Female"), n, replace = TRUE)),
  exposure_bin = rbinom(n, 1, 0.5)
)
dat$outcome <- rbinom(n, 1, plogis(-2 + 0.03*dat$age + 0.5*dat$exposure_bin))
## ---- Example 1: Binary exposure ----
fit_bin_exp<-final_prop_svyglm(dat, dep_var="outcome",
                  covariates=c("age","sex"),
                  exposure="exposure_bin",
                  id_var="psu", strata_var="strata",
                  weight_var="weight", outcome_covariates = NULL)
fit_bin_exp$OR_table
## ---- Example 2: Continuous exposure ----
fit_cont_exp <- final_prop_svyglm(
  dat,
  dep_var     = "outcome",
  covariates  = c("sex"),
  exposure    = "age",
  id_var      = "psu",
  strata_var  = "strata",
  weight_var  = "weight",
 exposure_type = "continuous",
 outcome_covariates = NULL)
fit_cont_exp$OR_table
#### ---- Example 1: Multinomial exposure ----
dat$exposure_3cat <- cut(dat$age,
breaks = quantile(dat$age, probs = c(0, 1/3, 2/3, 1)),  # tertiles
 labels = c("Low", "Medium", "High"),
include.lowest = TRUE)
# Numeric coding for exposure effect
exp_eff <- ifelse(dat$exposure_3cat == "Low", 0,
                ifelse(dat$exposure_3cat == "Medium", 0.6, 1.2))
dat$outcome <- rbinom(n,1,plogis(-3 +0.02 * dat$age +0.4  * (dat$sex == "Male") +exp_eff))
fit_multi_cat <- final_prop_svyglm(dat, dep_var     = "outcome",
covariates  = c("age", "sex"), exposure    = "exposure_3cat",
id_var      = "psu", strata_var  = "strata", weight_var  = "weight",
exposure_type = "multinomial",
outcome_covariates = NULL)
fit_multi_cat$OR_table
}
