#' Summarize Tipping Point Results (ARD Format)
#'
#' Creates a concise, analysis-results dataset (ARD) from a tipping point
#' analysis. Identifies the tipping point parameter where the upper CL of the hazard ratio crosses 1
#' and summarizes key metrics.
#'
#' @param object A `tipse` object returned by \link{tipping_point_model_free} or
#'   \link{tipping_point_model_based}.
#' @param ... Additional arguments not used.
#'
#' @return A data frame summarizing:
#'   \itemize{
#'   \item \code{HR} - hazard ratio at that tipping point
#'   \item \code{CONFINT} - 95% CI at tipping point
#'     \item \code{METHOD} - sampling type used
#'     \item \code{ARMIMP} - arm imputed
#'     \item \code{TIPPT} - parameter where upper CL first crosses 1
#'     \item \code{TIPUNIT} - parameter meaning
#'     \item \code{DESC} - textual interpretation
#'   }
#' @export
#'
#' @examples
#' cox1 <- survival::coxph(Surv(AVAL, EVENT) ~ TRT01P, data = codebreak200)
#' result <- tipping_point_model_based(
#'   dat = codebreak200,
#'   reason = "Early dropout",
#'   impute = "docetaxel",
#'   imputation_model = "weibull",
#'   J = 10,
#'   tipping_range = seq(0.1, 1, by = 0.05),
#'   cox_fit = cox1,
#'   verbose = TRUE,
#'   seed = 12345
#' )
#' summary(result)
summary.tipse <- function(object, ...) {
  #----------------------------#
  # Validation
  #----------------------------#
  sanitize_tipse(object)

  summary_results <- object$imputation_results
  km_data <- object$imputation_data
  dat <- object$original_data
  arm <- object$arm_to_impute
  method <- object$method_to_impute

  control <- levels(dat[["TRT01P"]])[1]
  trt <- levels(dat[["TRT01P"]])[2]

  #----------------------------#
  # Identify tipping point
  #----------------------------#
  tipping_param <- summary_results$parameter[summary_results$tipping_point]

  if (method == "random sampling") {
    tipping_unit <- paste(ifelse(arm == control, "best", "worst"), "percentile")
  } else if (method == "deterministic sampling") {
    tipping_unit <- paste("number of subjects", ifelse(arm == control, "extended censoring", "set as events"))
  } else if (method == "hazard inflation") {
    tipping_unit <- "% hazard inflation"
  } else if (method == "hazard deflation") {
    tipping_unit <- "% hazard deflation"
  } else {
    stop("Unsupported tipping point method: ", method)
  }

  if (length(tipping_param) == 0 || is.na(tipping_param)) {
    param_tipping <- NA
    hr_tipping <- NA
    ci_tipping <- NA
    interp <- "No tipping point detected (upper CL < 1 across all parameters)."
  } else {
    res_tp <- summary_results[summary_results$parameter == tipping_param, ]
    hr_tipping <- res_tp$HR
    ci_tipping <- sprintf("(%.4f-%.4f)", res_tp$HR_lowerCI, res_tp$HR_upperCI)
    param_tipping <- tipping_param
    if (method == "hazard inflation") {
      param_tipping <- (tipping_param - 1) * 100
    } else if (method == "hazard deflation") {
      param_tipping <- (1 - tipping_param) * 100
    }
    interp <- paste0(
      "Tipping point (upper CL \u2265 1) reached when imputing ", arm,
      " arm at ", param_tipping, " ", tipping_unit, "."
    )
  }

  #----------------------------#
  # Summarize table (ARD format)
  #----------------------------#
  summary_tbl <- data.frame(
    HR = round(hr_tipping, 4),
    CONFINT = ci_tipping,
    METHOD = method,
    ARMIMP = arm,
    TIPPT = param_tipping,
    TIPUNIT = tipping_unit,
    DESC = interp,
    stringsAsFactors = FALSE
  )

  return(summary_tbl)
}
