#' @title Install Package Dependencies
#' @description Installs Python dependencies (TensorFlow, Keras, Pandas) in a Conda environment for the transformerForecasting package.
#' @param env_name Character string specifying the Conda environment name (default: "r-reticulate").
#' @return Invisibly returns NULL after attempting to install dependencies.
#' @export
#' @examples
#' \dontrun{
#'   install_r_dependencies()
#' }
install_r_dependencies <- function(env_name = "r-reticulate") {

  # Step 1: Check if Python is already initialized
  if (reticulate::py_available()) {
    # Python is already loaded; check modules without switching environments
    installed_packages <- tryCatch(reticulate::py_list_packages()$package,
                                   error = function(e) character(0))
    required_packages <- c("tensorflow", "keras", "pandas")
    missing_packages <- setdiff(required_packages, installed_packages)

    if (length(missing_packages) > 0) {
      message("Installing missing Python packages: ", paste(missing_packages, collapse = ", "))
      reticulate::py_install(missing_packages, pip = TRUE)
    } else {
      message("All required Python packages are already installed in the active environment.")
    }
    return(invisible(NULL))
  }

  # Step 2: Check Python installation
  system_python <- tryCatch(Sys.which("python"), error = function(e) NULL)
  if (is.null(system_python) || !file.exists(system_python)) {
    warning("System Python not found! Attempting to install Miniconda.")
  } else {
    message("System Python detected at: ", system_python)
  }

  # Step 3: Check and install Conda if needed
  conda_path <- tryCatch(reticulate::conda_binary(), error = function(e) NULL)
  if (is.null(conda_path) || !file.exists(conda_path)) {
    message("Conda not found. Installing Miniconda...")
    tryCatch({
      reticulate::install_miniconda()
      conda_path <- reticulate::conda_binary()
    }, error = function(e) {
      warning("Miniconda installation failed! Please install it manually.")
      return(invisible(NULL))
    })
  }

  if (!is.null(conda_path) && file.exists(conda_path)) {
    message("Conda detected at: ", conda_path)
  } else {
    warning("Conda setup failed! Please install Miniconda manually.")
    return(invisible(NULL))
  }

  # Step 4: Ensure Conda environment exists
  if (!(env_name %in% reticulate::conda_list()$name)) {
    message("Conda environment not found. Creating one...")
    reticulate::conda_create(env_name)
  } else {
    message("Using existing Conda environment: ", env_name)
  }

  # Step 5: Set Conda environment (only if Python isn’t initialized)
  reticulate::use_condaenv(env_name, required = TRUE)

  # Step 6: Install required Python packages
  required_packages <- c("tensorflow", "keras", "pandas")
  installed_packages <- tryCatch(reticulate::py_list_packages()$package,
                                 error = function(e) character(0))
  missing_packages <- setdiff(required_packages, installed_packages)

  if (length(missing_packages) > 0) {
    message("Installing missing Python packages: ", paste(missing_packages, collapse = ", "))
    reticulate::conda_install(env_name, packages = missing_packages, pip = TRUE)
  } else {
    message("All required Python packages are already installed.")
  }

  message("Conda environment setup completed successfully!")
  invisible(NULL)
}
