#' uniScore
#'
#' @description
#' A function for unifying a person's scores from multiple tests
#'
#' @param scores a vector with a person's score from multiple tests
#' @param M Vector of means of the tests; must match the order of `scores`. Default: 0
#' @param SD Vector of standard deviations of the tests; must match the order of `scores`. Default: 1
#' @param b Vector of weightings for the test scores; must match the order of `scores`. Default: 1
#' @param corm a matrix that contains the correlations between the tests. There must be ones on the diagonal.
#' @param method the specification of the method for computing a unified score. "composite" (default) computes a non-latent unified score and "pca" computes the unified score on the principle component.
#'
#' @return
#'   \item{z}{A z score of the class `numeric` corresponding to the unified person's score; M = 0, SD = 1}
#'   \item{IQ}{An IQ score of the class `numeric` corresponding to the unified person's score; M = 100, SD = 15}
#'   \item{T}{A T score of the class `numeric` corresponding to the unified person's score; M = 50, SD = 10}
#'   \item{SW}{A SW score of the class `numeric` corresponding to the unified person's score; M = 100, SD = 10}
#'   \item{C}{A C score of the class `numeric` corresponding to the unified person's score; M = 5, SD = 2}
#'   \item{PR}{The percentile rank of the unified person's score of the class `numeric`}
#'
#' @examples
#' scores = c(110, 130, 120)
#' M = c(100, 100, 100)
#' SD = c(15, 15, 15)
#' b = c(1, 1, 1)
#' corm = matrix(c(1, 0.50, 0.35, 0.50, 1, 0.75, 0.35, 0.70, 1), nrow = 3, ncol = 3)
#' uniScore(scores, M, SD, b, corm, method = "composite")
#'
#' @export
uniScore <- function(scores, M = NULL, SD = NULL, b = NULL, corm, method = "composite") {
  num_scores = length(scores)
  if (is.null(M)) M = rep(0, num_scores)
  if (is.null(SD)) SD = rep(1, num_scores)
  if (is.null(b)) b = rep(1, num_scores)
  if (length(M) != num_scores || length(SD) != num_scores || length(b) != num_scores) {
    stop("scores, M, SD, and b must have the same length.")
  }
  if (!is.matrix(corm) || nrow(corm) != num_scores || ncol(corm) != num_scores) {
    stop("corm must be a square matrix with dimensions matching the length of scores.")
  }
  if (!method %in% c("composite", "pca")) {
    stop("Invalid method. Choose either 'composite' or 'pca'.")
  }
  if (method == "composite") {
    unified = composite(scores, M, SD, b, corm)
  } else if (method == "pca") {
    unified = pca(scores, M, SD, b, corm)
  }
  output = computeOutput(unified)
  return(output)
}
